//
// Cforall Version 1.0.0 Copyright (C) 2019 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Convert.cpp -- Convert between the new and old syntax trees.
//
// Author           : Thierry Delisle
// Created On       : Thu May 09 15::37::05 2019
// Last Modified By : Andrew Beach
// Last Modified On : Fri May 17 12:13:00 2019
// Update Count     : 3
//

#include "Convert.hpp"

#include "AST/Attribute.hpp"
#include "AST/Decl.hpp"
#include "AST/Expr.hpp"
#include "AST/Init.hpp"
#include "AST/Stmt.hpp"
#include "AST/TypeSubstitution.hpp"

#include "SynTree/Attribute.h"
#include "SynTree/Declaration.h"
#include "SynTree/TypeSubstitution.h"

//================================================================================================
// Utilities
template<template <class...> class C>
struct to {
	template<typename T>
	static auto from( T && v ) -> C< typename T::value_type > {
		C< typename T::value_type > l;
		std::move(std::begin(v), std::end(v), std::back_inserter(l));
		return l;
	}
};

//================================================================================================
class ConverterNewToOld : public ast::Visitor {
public:
	template<typename T>
	T * get() {
		T * ret = strict_dynamic_cast< T * >( node );
		node = nullptr;
		return ret;
	}

private:
	BaseSyntaxNode * node = nullptr;

	// All the visit functions:

	const ast::DeclWithType * visit( const ast::ObjectDecl * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::DeclWithType * visit( const ast::FunctionDecl * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Decl * visit( const ast::StructDecl * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Decl * visit( const ast::UnionDecl * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Decl * visit( const ast::EnumDecl * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Decl * visit( const ast::TraitDecl * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Decl * visit( const ast::TypeDecl * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Decl * visit( const ast::TypedefDecl * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::AsmDecl * visit( const ast::AsmDecl * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::StaticAssertDecl * visit( const ast::StaticAssertDecl * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::CompoundStmt * visit( const ast::CompoundStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::ExprStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::AsmStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::DirectiveStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::IfStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::WhileStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::ForStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::SwitchStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::CaseStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::BranchStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::ReturnStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::ThrowStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::TryStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::CatchStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::FinallyStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::WaitForStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::WithStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::NullStmt * visit( const ast::NullStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::DeclStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Stmt * visit( const ast::ImplicitCtorDtorStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ApplicationExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::NameExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::AddressExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::LabelAddressExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::CastExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::KeywordCastExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::VirtualCastExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedMemberExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::MemberExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::VariableExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ConstantExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::SizeofExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::AlignofExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedOffsetofExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::OffsetofExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::OffsetPackExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::LogicalExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ConditionalExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::CommaExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TypeExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::AsmExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ImplicitCopyCtorExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ConstructorExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::CompoundLiteralExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::RangeExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedTupleExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TupleExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TupleIndexExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TupleAssignExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::StmtExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UniqueExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedInitExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::InitExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::DeletedExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::DefaultArgExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::GenericExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::VoidType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::BasicType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::PointerType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::ArrayType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::ReferenceType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::QualifiedType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::FunctionType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::StructInstType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::UnionInstType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::EnumInstType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::TraitInstType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::TypeInstType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::TupleType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::TypeofType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::VarArgsType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::ZeroType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::OneType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::GlobalScopeType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Designation * visit( const ast::Designation * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Init * visit( const ast::SingleInit * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Init * visit( const ast::ListInit * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Init * visit( const ast::ConstructorInit * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Attribute * visit( const ast::Attribute * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::TypeSubstitution * visit( const ast::TypeSubstitution * node ) override final {
		(void)node;
		return nullptr;
	}
};

std::list< Declaration * > convert( std::list< ast::ptr< ast::Decl > > && translationUnit ) {
	ConverterNewToOld c;
	std::list< Declaration * > decls;
	for(auto d : translationUnit) {
		d->accept( c );
		decls.emplace_back( c.get<Declaration>() );
		delete d;
	}
	return decls;
}

//================================================================================================

class ConverterOldToNew : public Visitor {
public:
	ast::Decl * decl() {
		return strict_dynamic_cast< ast::Decl * >( node );
	}
private:
	ast::Node * node;

	// Local Utilities:

	template<typename NewT, typename OldT>
	NewT * getAccept1( OldT old ) {
		old->accept(*this);
		return strict_dynamic_cast< NewT * >( node );
	}

#	define GET_ACCEPT_1(child, type) \
		getAccept1< ast::type, decltype( old->child ) >( old->child )

	template<typename NewT, typename OldC>
	std::vector< ast::ptr<NewT> > getAcceptV( OldC& old ) {
		std::vector< ast::ptr<NewT> > ret;
		ret.reserve( old.size() );
		for ( auto a : old ) {
			a->accept( *this );
			ret.emplace_back( strict_dynamic_cast< NewT * >(node) );
		}
		return ret;
	}

#	define GET_ACCEPT_V(child, type) \
		getAcceptV< ast::type, decltype( old->child ) >( old->child )

	ast::Label make_label(Label* old) {
		return ast::Label(
			old->labelled->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute)
		);
	}

	template<template <class...> class C>
	C<ast::Label> make_labels(C<Label> olds) {
		C<ast::Label> ret;
		for (auto oldn : olds) {
			ret.push_back( make_label( &oldn ) );
		}
		return ret;
	}

#	define GET_LABELS_V(labels) \
		to<std::vector>::from( make_labels( std::move( labels ) ) )

	// Now all the visit functions:

	virtual void visit( ObjectDecl * old ) override final {
		auto decl = new ast::ObjectDecl(
			old->location,
			old->name,
			GET_ACCEPT_1(type, Type),
			GET_ACCEPT_1(init, Init),
			{ old->get_storageClasses().val },
			{ old->linkage.val },
			GET_ACCEPT_1(bitfieldWidth, Expr),
			GET_ACCEPT_V(attributes, Attribute),
			{ old->get_funcSpec().val }
		);
		decl->scopeLevel = old->scopeLevel;
		decl->mangleName = old->mangleName;
		decl->isDeleted  = old->isDeleted;
		decl->uniqueId   = old->uniqueId;
		decl->extension  = old->extension;

		this->node = decl;
	}

	virtual void visit( FunctionDecl * ) override final {

	}

	virtual void visit( StructDecl * old ) override final {
		auto decl = new ast::StructDecl(
			old->location,
			old->name,
			old->kind,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( UnionDecl * old ) override final {
		auto decl = new ast::UnionDecl(
			old->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( EnumDecl * old ) override final {
		auto decl = new ast::UnionDecl(
			old->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( TraitDecl * old ) override final {
		auto decl = new ast::UnionDecl(
			old->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( TypeDecl * ) override final {

	}

	virtual void visit( TypedefDecl * old ) override final {
		auto decl = new ast::TypedefDecl(
			old->location,
			old->name,
			{ old->storageClasses.val },
			GET_ACCEPT_1(base, Type),
			{ old->linkage.val }
		);
		decl->assertions = GET_ACCEPT_V(assertions, DeclWithType);
		decl->params     = GET_ACCEPT_V(parameters, TypeDecl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( AsmDecl * ) override final {

	}

	virtual void visit( StaticAssertDecl * ) override final {

	}

	virtual void visit( CompoundStmt * old ) override final {
		auto stmt = new ast::CompoundStmt(
			old->location,
			to<std::list>::from( GET_ACCEPT_V(kids, Stmt) ),
			GET_LABELS_V(old->labels)
		);

		this->node = stmt;
	}

	virtual void visit( ExprStmt * old ) override final {
		this->node = new ast::ExprStmt(
			old->location,
			GET_ACCEPT_1(expr, Expr),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( AsmStmt * old ) override final {
		this->node = new ast::AsmStmt(
			old->location,
			old->voltile,
			GET_ACCEPT_1(instruction, Expr),
			GET_ACCEPT_V(output, Expr),
			GET_ACCEPT_V(input, Expr),
			GET_ACCEPT_V(clobber, ConstantExpr),
			GET_LABELS_V(old->gotolabels),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( DirectiveStmt * old ) override final {
		this->node = new ast::DirectiveStmt(
			old->location,
			old->directive,
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( IfStmt * old ) override final {
		this->node = new ast::IfStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_1(thenPart, Stmt),
			GET_ACCEPT_1(elsePart, Stmt),
			GET_ACCEPT_V(initialization, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( SwitchStmt * old ) override final {
		this->node = new ast::SwitchStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_V(statements, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( CaseStmt * old ) override final {
		this->node = new ast::CaseStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_V(stmts, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( WhileStmt * old ) override final {
		this->node = new ast::WhileStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_1(body, Stmt),
			GET_ACCEPT_V(initialization, Stmt),
			old->isDoWhile,
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( ForStmt * old ) override final {
		this->node = new ast::ForStmt(
			old->location,
			GET_ACCEPT_V(initialization, Stmt),
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_1(increment, Expr),
			GET_ACCEPT_1(body, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( BranchStmt * old ) override final {
		if (old->computedTarget) {
			this->node = new ast::BranchStmt(
				old->location,
				GET_ACCEPT_1(computedTarget, Expr),
				GET_LABELS_V(old->labels)
			);
		} else {
			ast::BranchStmt::Kind kind;
			switch (old->type) {
			#define CASE(n) \
			case BranchStmt::n: \
				kind = ast::BranchStmt::n; \
				break
			CASE(Goto);
			CASE(Break);
			CASE(Continue);
			CASE(FallThrough);
			CASE(FallThroughDefault);
			#undef CASE
			default:
				assertf(false, "Invalid BranchStmt::Type %d\n", old->type);
			}

			Label label = old->originalTarget;
			auto stmt = new ast::BranchStmt(
				old->location,
				kind,
				make_label(&label),
				GET_LABELS_V(old->labels)
			);
			stmt->target = make_label(&old->target);
			this->node = stmt;
		}
	}

	virtual void visit( ReturnStmt * old ) override final {
		this->node = new ast::ReturnStmt(
			old->location,
			GET_ACCEPT_1(expr, Expr),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( ThrowStmt * old ) override final {
		ast::ThrowStmt::Kind kind;
		switch (old->kind) {
		case ThrowStmt::Terminate:
			kind = ast::ThrowStmt::Terminate;
			break;
		case ThrowStmt::Resume:
			kind = ast::ThrowStmt::Resume;
			break;
		default:
			assertf(false, "Invalid ThrowStmt::Kind %d\n", old->kind);
		}

		this->node = new ast::ThrowStmt(
			old->location,
			kind,
			GET_ACCEPT_1(expr, Expr),
			GET_ACCEPT_1(target, Expr),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( TryStmt * old ) override final {
		this->node = new ast::TryStmt(
			old->location,
			GET_ACCEPT_1(block, CompoundStmt),
			GET_ACCEPT_V(handlers, CatchStmt),
			GET_ACCEPT_1(finallyBlock, FinallyStmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( CatchStmt * old ) override final {
		ast::CatchStmt::Kind kind;
		switch (old->kind) {
		case CatchStmt::Terminate:
			kind = ast::CatchStmt::Terminate;
			break;
		case CatchStmt::Resume:
			kind = ast::CatchStmt::Resume;
			break;
		default:
			assertf(false, "Invalid CatchStmt::Kind %d\n", old->kind);
		}

		this->node = new ast::CatchStmt(
			old->location,
			kind,
			GET_ACCEPT_1(decl, Decl),
			GET_ACCEPT_1(cond, Expr),
			GET_ACCEPT_1(body, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( FinallyStmt * old ) override final {
		this->node = new ast::FinallyStmt(
			old->location,
			GET_ACCEPT_1(block, CompoundStmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( WaitForStmt * old ) override final {
		ast::WaitForStmt * stmt = new ast::WaitForStmt(
			old->location,
			GET_LABELS_V(old->labels)
		);

		stmt->clauses.reserve( old->clauses.size() );
		for (size_t i = 0 ; i < old->clauses.size() ; ++i) {
			stmt->clauses.push_back({
				ast::WaitForStmt::Target{
					GET_ACCEPT_1(clauses[i].target.function, Expr),
					GET_ACCEPT_V(clauses[i].target.arguments, Expr)
				},
				GET_ACCEPT_1(clauses[i].statement, Stmt),
				GET_ACCEPT_1(clauses[i].condition, Expr)
			});
		}
		stmt->timeout = {
			GET_ACCEPT_1(timeout.time, Expr),
			GET_ACCEPT_1(timeout.statement, Stmt),
			GET_ACCEPT_1(timeout.condition, Expr),
		};
		stmt->orElse = {
			GET_ACCEPT_1(timeout.statement, Stmt),
			GET_ACCEPT_1(timeout.condition, Expr),
		};

		this->node = stmt;
	}

	virtual void visit( WithStmt * old ) override final {
		this->node = new ast::WithStmt(
			old->location,
			GET_ACCEPT_V(exprs, Expr),
			GET_ACCEPT_1(stmt, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( NullStmt * old ) override final {
		this->node = new ast::NullStmt(
			old->location,
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( DeclStmt * old ) override final {
		this->node = new ast::DeclStmt(
			old->location,
			GET_ACCEPT_1(decl, Decl),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( ImplicitCtorDtorStmt * old ) override final {
		this->node = new ast::ImplicitCtorDtorStmt(
			old->location,
			GET_ACCEPT_1(callStmt, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	ast::TypeSubstitution * convertTypeSubstitution(const TypeSubstitution * old) {

		ast::TypeSubstitution *rslt = new ast::TypeSubstitution();

		for (decltype(old->begin()) old_i = old->begin(); old_i != old->end(); old_i++) {
			rslt->add( old_i->first,
			           getAccept1<ast::Type>(old_i->second) );
		}

		for (decltype(old->beginVar()) old_i = old->beginVar(); old_i != old->endVar(); old_i++) {
			rslt->addVar( old_i->first,
			              getAccept1<ast::Expr>(old_i->second) );
		}
	}

	void convertInferUnion(ast::Expr::InferUnion &nwInferred, InferredParams oldInferParams, const std::vector<UniqueId> &oldResnSlots) {
		
		(void) nwInferred;
		(void) oldInferParams;
		(void) oldResnSlots;
		
		// TODO
	}

	ast::Expr * visitBaseExpr(Expression * old, ast::Expr * nw) {

		nw->result = GET_ACCEPT_1(result, Type);
		nw->env    = convertTypeSubstitution(old->env);

		nw->extension = old->extension;
		convertInferUnion(nw->inferred, old->inferParams, old->resnSlots);

		return nw;
	}

	virtual void visit( ApplicationExpr * ) override final {
		// TODO
	}

	virtual void visit( UntypedExpr * ) override final {
		// TODO
	}

	virtual void visit( NameExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::NameExpr(
				old->location,
				old->get_name()
			)
		);
	}

	virtual void visit( CastExpr * ) override final {
		// TODO ... (rest)
	}

	virtual void visit( KeywordCastExpr * ) override final {

	}

	virtual void visit( VirtualCastExpr * ) override final {

	}

	virtual void visit( AddressExpr * ) override final {

	}

	virtual void visit( LabelAddressExpr * ) override final {

	}

	virtual void visit( UntypedMemberExpr * ) override final {

	}

	virtual void visit( MemberExpr * ) override final {

	}

	virtual void visit( VariableExpr * ) override final {

	}

	virtual void visit( ConstantExpr * ) override final {

	}

	virtual void visit( SizeofExpr * ) override final {

	}

	virtual void visit( AlignofExpr * ) override final {

	}

	virtual void visit( UntypedOffsetofExpr * ) override final {

	}

	virtual void visit( OffsetofExpr * ) override final {

	}

	virtual void visit( OffsetPackExpr * ) override final {

	}

	virtual void visit( LogicalExpr * ) override final {

	}

	virtual void visit( ConditionalExpr * ) override final {

	}

	virtual void visit( CommaExpr * ) override final {

	}

	virtual void visit( TypeExpr * ) override final {

	}

	virtual void visit( AsmExpr * ) override final {

	}

	virtual void visit( ImplicitCopyCtorExpr * ) override final {

	}

	virtual void visit( ConstructorExpr *  ) override final {

	}

	virtual void visit( CompoundLiteralExpr * ) override final {

	}

	virtual void visit( RangeExpr * ) override final {

	}

	virtual void visit( UntypedTupleExpr * ) override final {

	}

	virtual void visit( TupleExpr * ) override final {

	}

	virtual void visit( TupleIndexExpr * ) override final {

	}

	virtual void visit( TupleAssignExpr * ) override final {

	}

	virtual void visit( StmtExpr *  ) override final {

	}

	virtual void visit( UniqueExpr *  ) override final {

	}

	virtual void visit( UntypedInitExpr *  ) override final {

	}

	virtual void visit( InitExpr *  ) override final {

	}

	virtual void visit( DeletedExpr * ) override final {

	}

	virtual void visit( DefaultArgExpr * ) override final {

	}

	virtual void visit( GenericExpr * ) override final {

	}

	virtual void visit( VoidType * ) override final {

	}

	virtual void visit( BasicType * ) override final {

	}

	virtual void visit( PointerType * ) override final {

	}

	virtual void visit( ArrayType * ) override final {

	}

	virtual void visit( ReferenceType * ) override final {

	}

	virtual void visit( QualifiedType * ) override final {

	}

	virtual void visit( FunctionType * ) override final {

	}

	virtual void visit( StructInstType * ) override final {

	}

	virtual void visit( UnionInstType * ) override final {

	}

	virtual void visit( EnumInstType * ) override final {

	}

	virtual void visit( TraitInstType * ) override final {

	}

	virtual void visit( TypeInstType * ) override final {

	}

	virtual void visit( TupleType * ) override final {

	}

	virtual void visit( TypeofType * ) override final {

	}

	virtual void visit( AttrType * ) override final {

	}

	virtual void visit( VarArgsType * ) override final {

	}

	virtual void visit( ZeroType * ) override final {

	}

	virtual void visit( OneType * ) override final {

	}

	virtual void visit( GlobalScopeType * ) override final {

	}

	virtual void visit( Designation * ) override final {

	}

	virtual void visit( SingleInit * ) override final {

	}

	virtual void visit( ListInit * ) override final {

	}

	virtual void visit( ConstructorInit * ) override final {

	}

	virtual void visit( Constant * ) override final {

	}

	virtual void visit( Attribute * ) override final {

	}

	virtual void visit( AttrExpr * ) override final {

		assert( 0 );
	}
};

#undef GET_LABELS_V
#undef GET_ACCEPT_V
#undef GET_ACCEPT_1

std::list< ast::ptr< ast::Decl > > convert( const std::list< Declaration * > && translationUnit ) {
	ConverterOldToNew c;
	std::list< ast::ptr< ast::Decl > > decls;
	for(auto d : translationUnit) {
		d->accept( c );
		decls.emplace_back( c.decl() );
		delete d;
	}
	return decls;
}
