//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// TypedefTable.cc -- 
//
// Author           : Rodolfo G. Esteves
// Created On       : Sat May 16 15:20:13 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Apr 13 16:57:30 2016
// Update Count     : 24
//

#include <map>
#include <list>
#include <cassert>
#include "TypedefTable.h"
using namespace std;

#if 0
#include <iostream>
#define debugPrint( x ) cerr << x
#else
#define debugPrint( x )
#endif

TypedefTable::TypedefTable() : currentScope( 0 ) {}

bool TypedefTable::exists( const string &identifier ) {
	return table.count( identifier ) > 0;
}

int TypedefTable::isKind( const string &identifier ) const {
	tableType::const_iterator id_pos = table.find( identifier );
	// Name lookup defaults to identifier, and then the identifier's kind is set by the parser.
	if ( id_pos == table.end() ) return IDENTIFIER;
	return id_pos->second.begin()->kind;
}

void TypedefTable::changeKind( const string &identifier, kind_t kind ) {
	tableType::iterator id_pos = table.find( identifier );
	if ( id_pos == table.end() ) return;
	id_pos->second.begin()->kind = kind;
}

// SKULLDUGGERY: Generate a typedef for the aggregate name so the aggregate does not have to be qualified by
// "struct". Only generate the typedef, if the name is not in use. The typedef is implicitly (silently) removed
// if the name is explicitly used.
void TypedefTable::makeTypedef( const string &name ) {
	if ( ! typedefTable.exists( name ) ) {
		typedefTable.addToEnclosingScope( name, TypedefTable::TD );
	} // if
}

void TypedefTable::addToScope( const std::string &identifier, kind_t kind, int scope ) {
	if ( currentTrait != "" && scope == contextScope ) {
		DeferredEntry entry = { identifier, kind };
		contexts[currentTrait].push_back( entry );
	} else {
		debugPrint( "Adding " << identifier << " as kind " << kind << " scope " << scope << " from scope " << currentScope << endl );
		Entry newEntry = { scope, kind };
		tableType::iterator curPos = table.find( identifier );
		if ( curPos == table.end()) {
			list<Entry> newList;
			newList.push_front( newEntry );
			table[identifier] = newList;
		} else {
			list<Entry>::iterator listPos = (*curPos ).second.begin();
			while ( listPos != (*curPos ).second.end() && listPos->scope > scope ) {
				listPos++;
			} // while
			(*curPos ).second.insert( listPos, newEntry );
		} // if
	} // if
}

void TypedefTable::addToCurrentScope( const std::string &identifier, kind_t kind ) {
	addToScope( identifier, kind, currentScope );
}

void TypedefTable::addToCurrentScope( kind_t kind ) {
	addToCurrentScope( nextIdentifiers.top(), kind );
}

void TypedefTable::addToEnclosingScope( const std::string &identifier, kind_t kind ) {
	assert( currentScope >= 1 );
	addToScope( identifier, kind, currentScope - 1 );
}

void TypedefTable::addToEnclosingScope( kind_t kind ) {
	addToEnclosingScope( nextIdentifiers.top(), kind );
}

void TypedefTable::addToEnclosingScope2( const std::string &identifier, kind_t kind ) {
	assert( currentScope >= 2 );
	addToScope( identifier, kind, currentScope - 2 );
}

void TypedefTable::addToEnclosingScope2( kind_t kind ) {
	addToEnclosingScope2( nextIdentifiers.top(), kind );
}

void TypedefTable::setNextIdentifier( const std::string &identifier ) {
	nextIdentifiers.top() = identifier;
}

void TypedefTable::openTrait( const std::string &contextName ) {
	map< string, deferListType >::iterator i = contexts.find( contextName );
	if ( i != contexts.end() ) {
		deferListType &entries = i->second;
		for ( deferListType::iterator i = entries.begin(); i != entries.end(); i++) {
			addToEnclosingScope( i->identifier, i->kind );
		} // for
	} // if
}

void TypedefTable::enterScope() {
	currentScope += 1;
	deferListStack.push( deferListType() );
	nextIdentifiers.push( "" );
	debugPrint( "Entering scope " << currentScope << ", nextIdentifiers size is " << nextIdentifiers.size() << endl );
}

void TypedefTable::leaveScope() {
	debugPrint( "Leaving scope " << currentScope << endl );
	for ( tableType::iterator i = table.begin(); i != table.end(); ) {
		list<Entry> &declList = (*i).second;
		while ( ! declList.empty() && declList.front().scope == currentScope ) {
			declList.pop_front();
		}
		if ( declList.empty() ) {						// standard idom for erasing during traversal
			table.erase( i++ );
		} else
			++i;
	} // for
	currentScope -= 1;
	for ( deferListType::iterator i = deferListStack.top().begin(); i != deferListStack.top().end(); i++ ) {
		addToCurrentScope( i->identifier, i->kind );
	} // for
	deferListStack.pop();
	debugPrint( "nextIdentifiers size is " << nextIdentifiers.size() << " top is " << nextIdentifiers.top() << endl );
	nextIdentifiers.pop();
}

void TypedefTable::enterTrait( const std::string &contextName ) {
	currentTrait = contextName;
	contextScope = currentScope;
}

void TypedefTable::leaveTrait() {
	currentTrait = "";
}

void TypedefTable::print( void ) const {
	for ( tableType::const_iterator i = table.begin(); i != table.end(); i++) {
		debugPrint( (*i ).first << ": " );
		list<Entry> declList = (*i).second;
		for ( list<Entry>::const_iterator j = declList.begin(); j != declList.end(); j++ ) {
			debugPrint( "(" << (*j).scope << " " << (*j).kind << ") " );
		}
		debugPrint( endl );
	} // for
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
