//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Declaration.h --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Mar  7 07:40:42 2017
// Update Count     : 113
//

#ifndef DECLARATION_H
#define DECLARATION_H

#include <string>

#include "BaseSyntaxNode.h"
#include "Mutator.h"
#include "Visitor.h"
#include "SynTree.h"
#include "Parser/LinkageSpec.h"
#include "Parser/ParseNode.h"

class Declaration : public BaseSyntaxNode {
  public:
	Declaration( const std::string &name, DeclarationNode::StorageClasses scs, LinkageSpec::Spec linkage );
	Declaration( const Declaration &other );
	virtual ~Declaration();

	const std::string &get_name() const { return name; }
	void set_name( std::string newValue ) { name = newValue; }

	DeclarationNode::StorageClasses get_storageClasses() const { return storageClasses; }

	LinkageSpec::Spec get_linkage() const { return linkage; }
	void set_linkage( LinkageSpec::Spec newValue ) { linkage = newValue; }

	UniqueId get_uniqueId() const { return uniqueId; }

	bool get_extension() const { return extension; }
	Declaration *set_extension( bool exten ) { extension = exten; return this; }

	void fixUniqueId( void );
	virtual Declaration *clone() const = 0;
	virtual void accept( Visitor &v ) = 0;
	virtual Declaration *acceptMutator( Mutator &m ) = 0;
	virtual void print( std::ostream &os, int indent = 0 ) const = 0;
	virtual void printShort( std::ostream &os, int indent = 0 ) const = 0;

	static void dumpIds( std::ostream &os );
	static Declaration *declFromId( UniqueId id );
  private:
	std::string name;
	DeclarationNode::StorageClasses storageClasses;
	LinkageSpec::Spec linkage;
	UniqueId uniqueId;
	bool extension = false;
};

class DeclarationWithType : public Declaration {
  public:
	DeclarationWithType( const std::string &name, DeclarationNode::StorageClasses scs, LinkageSpec::Spec linkage, const std::list< Attribute * > & attributes, DeclarationNode::FuncSpecifiers fs );
	DeclarationWithType( const DeclarationWithType &other );
	virtual ~DeclarationWithType();

	std::string get_mangleName() const { return mangleName; }
	DeclarationWithType * set_mangleName( std::string newValue ) { mangleName = newValue; return this; }

	std::string get_scopedMangleName() const { return mangleName + "_" + std::to_string(scopeLevel); }

	int get_scopeLevel() const { return scopeLevel; }
	DeclarationWithType * set_scopeLevel( int newValue ) { scopeLevel = newValue; return this; }

	ConstantExpr *get_asmName() const { return asmName; }
	DeclarationWithType * set_asmName( ConstantExpr *newValue ) { asmName = newValue; return this; }

	std::list< Attribute * >& get_attributes() { return attributes; }
	const std::list< Attribute * >& get_attributes() const { return attributes; }

	DeclarationNode::FuncSpecifiers get_funcSpec() const { return fs; }
	//void set_functionSpecifiers( DeclarationNode::FuncSpecifiers newValue ) { fs = newValue; }

	virtual DeclarationWithType *clone() const = 0;
	virtual DeclarationWithType *acceptMutator( Mutator &m ) = 0;

	virtual Type *get_type() const = 0;
	virtual void set_type(Type *) = 0;
  private:
	// this represents the type with all types and typedefs expanded it is generated by SymTab::Validate::Pass2
	std::string mangleName;
	// need to remember the scope level at which the variable was declared, so that shadowed identifiers can be accessed
	int scopeLevel = 0;

	ConstantExpr *asmName;
	std::list< Attribute * > attributes;
	DeclarationNode::FuncSpecifiers fs;
};

class ObjectDecl : public DeclarationWithType {
	typedef DeclarationWithType Parent;
  public:
	ObjectDecl( const std::string &name, DeclarationNode::StorageClasses scs, LinkageSpec::Spec linkage, Expression *bitfieldWidth, Type *type, Initializer *init,
				const std::list< Attribute * > attributes = std::list< Attribute * >(), DeclarationNode::FuncSpecifiers fs = DeclarationNode::FuncSpecifiers() );
	ObjectDecl( const ObjectDecl &other );
	virtual ~ObjectDecl();

	virtual Type *get_type() const { return type; }
	virtual void set_type(Type *newType) { type = newType; }

	Initializer *get_init() const { return init; }
	void set_init( Initializer *newValue ) { init = newValue; }

	Expression *get_bitfieldWidth() const { return bitfieldWidth; }
	void set_bitfieldWidth( Expression *newValue ) { bitfieldWidth = newValue; }

	virtual ObjectDecl *clone() const { return new ObjectDecl( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual DeclarationWithType *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
	virtual void printShort( std::ostream &os, int indent = 0 ) const;
  private:
	Type *type;
	Initializer *init;
	Expression *bitfieldWidth;
};

class FunctionDecl : public DeclarationWithType {
	typedef DeclarationWithType Parent;
  public:
	FunctionDecl( const std::string &name, DeclarationNode::StorageClasses scs, LinkageSpec::Spec linkage, FunctionType *type, CompoundStmt *statements,
				  const std::list< Attribute * > attributes = std::list< Attribute * >(), DeclarationNode::FuncSpecifiers fs = DeclarationNode::FuncSpecifiers() );
	FunctionDecl( const FunctionDecl &other );
	virtual ~FunctionDecl();

	Type *get_type() const;
	virtual void set_type(Type *);

	FunctionType *get_functionType() const { return type; }
	void set_functionType( FunctionType *newValue ) { type = newValue; }
	CompoundStmt *get_statements() const { return statements; }
	void set_statements( CompoundStmt *newValue ) { statements = newValue; }

	virtual FunctionDecl *clone() const { return new FunctionDecl( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual DeclarationWithType *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
	virtual void printShort( std::ostream &os, int indent = 0 ) const;
  private:
	FunctionType *type;
	CompoundStmt *statements;
};

class NamedTypeDecl : public Declaration {
	typedef Declaration Parent;
  public:
	NamedTypeDecl( const std::string &name, DeclarationNode::StorageClasses scs, Type *type );
	NamedTypeDecl( const NamedTypeDecl &other );
	virtual ~NamedTypeDecl();

	Type *get_base() const { return base; }
	void set_base( Type *newValue ) { base = newValue; }
	std::list< TypeDecl* >& get_parameters() { return parameters; }
	std::list< DeclarationWithType* >& get_assertions() { return assertions; }

	virtual NamedTypeDecl *clone() const = 0;
	virtual void print( std::ostream &os, int indent = 0 ) const;
	virtual void printShort( std::ostream &os, int indent = 0 ) const;
  protected:
	virtual std::string typeString() const = 0;
  private:
	Type *base;
	std::list< TypeDecl* > parameters;
	std::list< DeclarationWithType* > assertions;
};

class TypeDecl : public NamedTypeDecl {
	typedef NamedTypeDecl Parent;
  public:
	enum Kind { Any, Dtype, Ftype, Ttype };
	/// Data extracted from a type decl
	struct Data {
		TypeDecl::Kind kind;
		bool isComplete;
		Data() : kind( (TypeDecl::Kind)-1 ), isComplete( false ) {}
		Data( TypeDecl * typeDecl ) : Data( typeDecl->get_kind(), typeDecl->isComplete() ) {}
		Data( Kind kind, bool isComplete ) : kind( kind ), isComplete( isComplete ) {}
		bool operator==(const Data & other) const { return kind == other.kind && isComplete == other.isComplete; }
		bool operator!=(const Data & other) const { return !(*this == other);}
	};

	TypeDecl( const std::string &name, DeclarationNode::StorageClasses scs, Type *type, Kind kind );
	TypeDecl( const TypeDecl &other );

	Kind get_kind() const { return kind; }

	bool isComplete() const { return kind == Any || sized; }
	bool get_sized() const { return sized; }
	TypeDecl * set_sized( bool newValue ) { sized = newValue; return this; }

	virtual TypeDecl *clone() const { return new TypeDecl( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual TypeDecl *acceptMutator( Mutator &m ) { return m.mutate( this ); }
  private:
	virtual std::string typeString() const;
	Kind kind;
	bool sized;
};

class TypedefDecl : public NamedTypeDecl {
	typedef NamedTypeDecl Parent;
  public:
	TypedefDecl( const std::string &name, DeclarationNode::StorageClasses scs, Type *type ) : Parent( name, scs, type ) {}
	TypedefDecl( const TypedefDecl &other ) : Parent( other ) {}

	virtual TypedefDecl *clone() const { return new TypedefDecl( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Declaration *acceptMutator( Mutator &m ) { return m.mutate( this ); }
  private:
	virtual std::string typeString() const;
};

class AggregateDecl : public Declaration {
	typedef Declaration Parent;
  public:
	AggregateDecl( const std::string &name, const std::list< Attribute * > & attributes = std::list< class Attribute * >() );
	AggregateDecl( const AggregateDecl &other );
	virtual ~AggregateDecl();

	std::list<Declaration*>& get_members() { return members; }
	std::list<TypeDecl*>& get_parameters() { return parameters; }

	std::list< Attribute * >& get_attributes() { return attributes; }
	const std::list< Attribute * >& get_attributes() const { return attributes; }

	bool has_body() const { return body; }
	AggregateDecl * set_body( bool body ) { AggregateDecl::body = body; return this; }

	virtual void print( std::ostream &os, int indent = 0 ) const;
	virtual void printShort( std::ostream &os, int indent = 0 ) const;
  protected:
	virtual std::string typeString() const = 0;

  private:
	std::list<Declaration*> members;
	std::list<TypeDecl*> parameters;
	bool body;
	std::list< Attribute * > attributes;
};

class StructDecl : public AggregateDecl {
	typedef AggregateDecl Parent;
  public:
	StructDecl( const std::string &name, const std::list< Attribute * > & attributes = std::list< class Attribute * >() ) : Parent( name, attributes ) {}
	StructDecl( const StructDecl &other ) : Parent( other ) {}

	virtual StructDecl *clone() const { return new StructDecl( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Declaration *acceptMutator( Mutator &m ) { return m.mutate( this ); }
  private:
	virtual std::string typeString() const;
};

class UnionDecl : public AggregateDecl {
	typedef AggregateDecl Parent;
  public:
	UnionDecl( const std::string &name, const std::list< Attribute * > & attributes = std::list< class Attribute * >() ) : Parent( name, attributes ) {}
	UnionDecl( const UnionDecl &other ) : Parent( other ) {}

	virtual UnionDecl *clone() const { return new UnionDecl( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Declaration *acceptMutator( Mutator &m ) { return m.mutate( this ); }
  private:
	virtual std::string typeString() const;
};

class EnumDecl : public AggregateDecl {
	typedef AggregateDecl Parent;
  public:
	EnumDecl( const std::string &name, const std::list< Attribute * > & attributes = std::list< class Attribute * >() ) : Parent( name, attributes ) {}
	EnumDecl( const EnumDecl &other ) : Parent( other ) {}

	virtual EnumDecl *clone() const { return new EnumDecl( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Declaration *acceptMutator( Mutator &m ) { return m.mutate( this ); }
  private:
	virtual std::string typeString() const;
};

class TraitDecl : public AggregateDecl {
	typedef AggregateDecl Parent;
  public:
	TraitDecl( const std::string &name, const std::list< Attribute * > & attributes ) : Parent( name ) {
		assertf( attributes.empty(), "attribute unsupported for traits" );
	}
	TraitDecl( const TraitDecl &other ) : Parent( other ) {}

	virtual TraitDecl *clone() const { return new TraitDecl( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Declaration *acceptMutator( Mutator &m ) { return m.mutate( this ); }
  private:
	virtual std::string typeString() const;
};

class AsmDecl : public Declaration {
  public:
	AsmDecl( AsmStmt *stmt );
	AsmDecl( const AsmDecl &other );
	virtual ~AsmDecl();

	AsmStmt *get_stmt() { return stmt; }
	void set_stmt( AsmStmt *newValue ) { stmt = newValue; }

	virtual AsmDecl *clone() const { return new AsmDecl( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual AsmDecl *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
	virtual void printShort( std::ostream &os, int indent = 0 ) const;
  private:
	AsmStmt *stmt;
};

std::ostream & operator<<( std::ostream & out, const Declaration * decl );
std::ostream & operator<<( std::ostream & os, const TypeDecl::Data & data );

#endif // DECLARATION_H

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
