//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Resolver.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 12:17:01 2015
// Last Modified By : Andrew Beach
// Last Modified On : Tus Aug  8 16:06:00 2017
// Update Count     : 212
//

#include <stddef.h>                      // for NULL
#include <cassert>                       // for strict_dynamic_cast, assert
#include <memory>                        // for allocator, allocator_traits<...
#include <tuple>                         // for get

#include "Alternative.h"                 // for Alternative, AltList
#include "AlternativeFinder.h"           // for AlternativeFinder, resolveIn...
#include "Common/PassVisitor.h"          // for PassVisitor
#include "Common/SemanticError.h"        // for SemanticError
#include "Common/utility.h"              // for ValueGuard, group_iterate
#include "CurrentObject.h"               // for CurrentObject
#include "InitTweak/InitTweak.h"         // for isIntrinsicSingleArgCallStmt
#include "RenameVars.h"                  // for RenameVars, global_renamer
#include "ResolvExpr/TypeEnvironment.h"  // for TypeEnvironment
#include "ResolveTypeof.h"               // for resolveTypeof
#include "Resolver.h"
#include "SymTab/Autogen.h"              // for SizeType
#include "SymTab/Indexer.h"              // for Indexer
#include "SynTree/Declaration.h"         // for ObjectDecl, TypeDecl, Declar...
#include "SynTree/Expression.h"          // for Expression, CastExpr, InitExpr
#include "SynTree/Initializer.h"         // for ConstructorInit, SingleInit
#include "SynTree/Statement.h"           // for ForStmt, Statement, BranchStmt
#include "SynTree/Type.h"                // for Type, BasicType, PointerType
#include "SynTree/TypeSubstitution.h"    // for TypeSubstitution
#include "SynTree/Visitor.h"             // for acceptAll, maybeAccept
#include "typeops.h"                     // for extractResultType
#include "Unify.h"                       // for unify

using namespace std;

namespace ResolvExpr {
	struct Resolver final : public WithIndexer, public WithGuards, public WithVisitorRef<Resolver>, public WithShortCircuiting {
		Resolver() {}
		Resolver( const SymTab::Indexer & other ) {
			indexer = other;
		}

		void previsit( FunctionDecl *functionDecl );
		void postvisit( FunctionDecl *functionDecl );
		void previsit( ObjectDecl *objectDecll );
		void previsit( TypeDecl *typeDecl );
		void previsit( EnumDecl * enumDecl );

		void previsit( ArrayType * at );
		void previsit( PointerType * at );

		void previsit( ExprStmt *exprStmt );
		void previsit( AsmExpr *asmExpr );
		void previsit( AsmStmt *asmStmt );
		void previsit( IfStmt *ifStmt );
		void previsit( WhileStmt *whileStmt );
		void previsit( ForStmt *forStmt );
		void previsit( SwitchStmt *switchStmt );
		void previsit( CaseStmt *caseStmt );
		void previsit( BranchStmt *branchStmt );
		void previsit( ReturnStmt *returnStmt );
		void previsit( ThrowStmt *throwStmt );
		void previsit( CatchStmt *catchStmt );
		void previsit( WaitForStmt * stmt );

		void previsit( SingleInit *singleInit );
		void previsit( ListInit *listInit );
		void previsit( ConstructorInit *ctorInit );
	  private:
  	typedef std::list< Initializer * >::iterator InitIterator;

		template< typename PtrType >
		void handlePtrType( PtrType * type );

	  void resolveAggrInit( ReferenceToType *, InitIterator &, InitIterator & );
	  void resolveSingleAggrInit( Declaration *, InitIterator &, InitIterator &, TypeSubstitution sub );
	  void fallbackInit( ConstructorInit * ctorInit );

		Type * functionReturn = nullptr;
		CurrentObject currentObject = nullptr;
		bool inEnumDecl = false;
	};

	void resolve( std::list< Declaration * > translationUnit ) {
		PassVisitor<Resolver> resolver;
		acceptAll( translationUnit, resolver );
	}

	void resolveDecl( Declaration * decl, const SymTab::Indexer &indexer ) {
		PassVisitor<Resolver> resolver( indexer );
		maybeAccept( decl, resolver );
	}

	// used in resolveTypeof
	Expression *resolveInVoidContext( Expression *expr, const SymTab::Indexer &indexer ) {
		TypeEnvironment env;
		return resolveInVoidContext( expr, indexer, env );
	}

	namespace {
		void finishExpr( Expression *expr, const TypeEnvironment &env, TypeSubstitution * oldenv = nullptr ) {
			expr->env = oldenv ? oldenv->clone() : new TypeSubstitution;
			env.makeSubstitution( *expr->get_env() );
		}
	} // namespace

	void findVoidExpression( Expression *& untyped, const SymTab::Indexer &indexer ) {
		global_renamer.reset();
		TypeEnvironment env;
		Expression *newExpr = resolveInVoidContext( untyped, indexer, env );
		finishExpr( newExpr, env, untyped->env );
		delete untyped;
		untyped = newExpr;
	}

	void findSingleExpression( Expression *&untyped, const SymTab::Indexer &indexer ) {
		if ( ! untyped ) return;
		TypeEnvironment env;
		AlternativeFinder finder( indexer, env );
		finder.find( untyped );
		#if 0
		if ( finder.get_alternatives().size() != 1 ) {
			std::cerr << "untyped expr is ";
			untyped->print( std::cerr );
			std::cerr << std::endl << "alternatives are:";
			for ( const Alternative & alt : finder.get_alternatives() ) {
				alt.print( std::cerr );
			} // for
		} // if
		#endif
		assertf( finder.get_alternatives().size() == 1, "findSingleExpression: must have exactly one alternative at the end." );
		Alternative &choice = finder.get_alternatives().front();
		Expression *newExpr = choice.expr->clone();
		finishExpr( newExpr, choice.env, untyped->env );
		delete untyped;
		untyped = newExpr;
	}

	void findSingleExpression( Expression *& untyped, Type * type, const SymTab::Indexer & indexer ) {
		assert( untyped && type );
		untyped = new CastExpr( untyped, type );
		findSingleExpression( untyped, indexer );
		if ( CastExpr * castExpr = dynamic_cast< CastExpr * >( untyped ) ) {
			if ( ResolvExpr::typesCompatible( castExpr->arg->result, castExpr->result, indexer ) ) {
				// cast is to the same type as its argument, so it's unnecessary -- remove it
				untyped = castExpr->arg;
				castExpr->arg = nullptr;
				delete castExpr;
			}
		}
	}

	namespace {
		bool isIntegralType( Type *type ) {
			if ( dynamic_cast< EnumInstType * >( type ) ) {
				return true;
			} else if ( BasicType *bt = dynamic_cast< BasicType * >( type ) ) {
				return bt->isInteger();
			} else if ( dynamic_cast< ZeroType* >( type ) != nullptr || dynamic_cast< OneType* >( type ) != nullptr ) {
				return true;
			} else {
				return false;
			} // if
		}

		void findIntegralExpression( Expression *& untyped, const SymTab::Indexer &indexer ) {
			TypeEnvironment env;
			AlternativeFinder finder( indexer, env );
			finder.find( untyped );
#if 0
			if ( finder.get_alternatives().size() != 1 ) {
				std::cout << "untyped expr is ";
				untyped->print( std::cout );
				std::cout << std::endl << "alternatives are:";
				for ( std::list< Alternative >::const_iterator i = finder.get_alternatives().begin(); i != finder.get_alternatives().end(); ++i ) {
					i->print( std::cout );
				} // for
			} // if
#endif
			Expression *newExpr = 0;
			const TypeEnvironment *newEnv = 0;
			for ( AltList::const_iterator i = finder.get_alternatives().begin(); i != finder.get_alternatives().end(); ++i ) {
				if ( i->expr->get_result()->size() == 1 && isIntegralType( i->expr->get_result() ) ) {
					if ( newExpr ) {
						throw SemanticError( "Too many interpretations for case control expression", untyped );
					} else {
						newExpr = i->expr->clone();
						newEnv = &i->env;
					} // if
				} // if
			} // for
			if ( ! newExpr ) {
				throw SemanticError( "No interpretations for case control expression", untyped );
			} // if
			finishExpr( newExpr, *newEnv, untyped->env );
			delete untyped;
			untyped = newExpr;
		}

	}

	void Resolver::previsit( ObjectDecl *objectDecl ) {
		Type *new_type = resolveTypeof( objectDecl->get_type(), indexer );
		objectDecl->set_type( new_type );
		// To handle initialization of routine pointers, e.g., int (*fp)(int) = foo(), means that class-variable
		// initContext is changed multiple time because the LHS is analysed twice. The second analysis changes
		// initContext because of a function type can contain object declarations in the return and parameter types. So
		// each value of initContext is retained, so the type on the first analysis is preserved and used for selecting
		// the RHS.
		GuardValue( currentObject );
		currentObject = CurrentObject( objectDecl->get_type() );
		if ( inEnumDecl && dynamic_cast< EnumInstType * >( objectDecl->get_type() ) ) {
			// enumerator initializers should not use the enum type to initialize, since
			// the enum type is still incomplete at this point. Use signed int instead.
			currentObject = CurrentObject( new BasicType( Type::Qualifiers(), BasicType::SignedInt ) );
		}
	}

	template< typename PtrType >
	void Resolver::handlePtrType( PtrType * type ) {
		if ( type->get_dimension() ) {
			findSingleExpression( type->dimension, SymTab::SizeType->clone(), indexer );
		}
	}

	void Resolver::previsit( ArrayType * at ) {
		handlePtrType( at );
	}

	void Resolver::previsit( PointerType * pt ) {
		handlePtrType( pt );
	}

	void Resolver::previsit( TypeDecl *typeDecl ) {
		if ( typeDecl->get_base() ) {
			Type *new_type = resolveTypeof( typeDecl->get_base(), indexer );
			typeDecl->set_base( new_type );
		} // if
	}

	void Resolver::previsit( FunctionDecl *functionDecl ) {
#if 0
		std::cerr << "resolver visiting functiondecl ";
		functionDecl->print( std::cerr );
		std::cerr << std::endl;
#endif
		Type *new_type = resolveTypeof( functionDecl->get_type(), indexer );
		functionDecl->set_type( new_type );
		GuardValue( functionReturn );
		functionReturn = ResolvExpr::extractResultType( functionDecl->get_functionType() );
	}

	void Resolver::postvisit( FunctionDecl *functionDecl ) {
		// default value expressions have an environment which shouldn't be there and trips up later passes.
		// xxx - it might be necessary to somehow keep the information from this environment, but I can't currently
		// see how it's useful.
		for ( Declaration * d : functionDecl->get_functionType()->get_parameters() ) {
			if ( ObjectDecl * obj = dynamic_cast< ObjectDecl * >( d ) ) {
				if ( SingleInit * init = dynamic_cast< SingleInit * >( obj->get_init() ) ) {
					delete init->get_value()->get_env();
					init->get_value()->set_env( nullptr );
				}
			}
		}
	}

	void Resolver::previsit( EnumDecl * ) {
		// in case we decide to allow nested enums
		GuardValue( inEnumDecl );
		inEnumDecl = true;
	}

	void Resolver::previsit( ExprStmt *exprStmt ) {
		visit_children = false;
		assertf( exprStmt->expr, "ExprStmt has null Expression in resolver" );
		findVoidExpression( exprStmt->expr, indexer );
	}

	void Resolver::previsit( AsmExpr *asmExpr ) {
		visit_children = false;
		findVoidExpression( asmExpr->operand, indexer );
		if ( asmExpr->get_inout() ) {
			findVoidExpression( asmExpr->inout, indexer );
		} // if
	}

	void Resolver::previsit( AsmStmt *asmStmt ) {
		visit_children = false;
		acceptAll( asmStmt->get_input(), *visitor );
		acceptAll( asmStmt->get_output(), *visitor );
	}

	void Resolver::previsit( IfStmt *ifStmt ) {
		findSingleExpression( ifStmt->condition, indexer );
	}

	void Resolver::previsit( WhileStmt *whileStmt ) {
		findSingleExpression( whileStmt->condition, indexer );
	}

	void Resolver::previsit( ForStmt *forStmt ) {
		if ( forStmt->condition ) {
			findSingleExpression( forStmt->condition, indexer );
		} // if

		if ( forStmt->increment ) {
			findVoidExpression( forStmt->increment, indexer );
		} // if
	}

	void Resolver::previsit( SwitchStmt *switchStmt ) {
		GuardValue( currentObject );
		findIntegralExpression( switchStmt->condition, indexer );

		currentObject = CurrentObject( switchStmt->condition->result );
	}

	void Resolver::previsit( CaseStmt *caseStmt ) {
		if ( caseStmt->get_condition() ) {
			std::list< InitAlternative > initAlts = currentObject.getOptions();
			assertf( initAlts.size() == 1, "SwitchStmt did not correctly resolve an integral expression." );
			// must remove cast from case statement because RangeExpr cannot be cast.
			Expression * newExpr = new CastExpr( caseStmt->condition, initAlts.front().type->clone() );
			findSingleExpression( newExpr, indexer );
			CastExpr * castExpr = strict_dynamic_cast< CastExpr * >( newExpr );
			caseStmt->condition = castExpr->arg;
			castExpr->arg = nullptr;
			delete castExpr;
		}
	}

	void Resolver::previsit( BranchStmt *branchStmt ) {
		visit_children = false;
		// must resolve the argument for a computed goto
		if ( branchStmt->get_type() == BranchStmt::Goto ) { // check for computed goto statement
			if ( branchStmt->computedTarget ) {
				// computed goto argument is void *
				findSingleExpression( branchStmt->computedTarget, new PointerType( Type::Qualifiers(), new VoidType( Type::Qualifiers() ) ), indexer );
			} // if
		} // if
	}

	void Resolver::previsit( ReturnStmt *returnStmt ) {
		visit_children = false;
		if ( returnStmt->expr ) {
			findSingleExpression( returnStmt->expr, functionReturn->clone(), indexer );
		} // if
	}

	void Resolver::previsit( ThrowStmt *throwStmt ) {
		visit_children = false;
		// TODO: Replace *exception type with &exception type.
		if ( throwStmt->get_expr() ) {
			StructDecl * exception_decl =
				indexer.lookupStruct( "__cfaehm__base_exception_t" );
			assert( exception_decl );
			Type * exceptType = new PointerType( noQualifiers, new StructInstType( noQualifiers, exception_decl ) );
			findSingleExpression( throwStmt->expr, exceptType, indexer );
		}
	}

	void Resolver::previsit( CatchStmt *catchStmt ) {
		if ( catchStmt->cond ) {
			findSingleExpression( catchStmt->cond, new BasicType( noQualifiers, BasicType::Bool ), indexer );
		}
	}

	template< typename iterator_t >
	inline bool advance_to_mutex( iterator_t & it, const iterator_t & end ) {
		while( it != end && !(*it)->get_type()->get_mutex() ) {
			it++;
		}

		return it != end;
	}

	void Resolver::previsit( WaitForStmt * stmt ) {
		visit_children = false;

		// Resolve all clauses first
		for( auto& clause : stmt->clauses ) {

			TypeEnvironment env;
			AlternativeFinder funcFinder( indexer, env );

			// Find all alternatives for a function in canonical form
			funcFinder.findWithAdjustment( clause.target.function );

			if ( funcFinder.get_alternatives().empty() ) {
				stringstream ss;
				ss << "Use of undeclared indentifier '";
				ss << strict_dynamic_cast<NameExpr*>( clause.target.function )->name;
				ss << "' in call to waitfor";
				throw SemanticError( ss.str() );
			}

			// Find all alternatives for all arguments in canonical form
			std::list< AlternativeFinder > argAlternatives;
			funcFinder.findSubExprs( clause.target.arguments.begin(), clause.target.arguments.end(), back_inserter( argAlternatives ) );

			// List all combinations of arguments
			std::list< AltList > possibilities;
			combos( argAlternatives.begin(), argAlternatives.end(), back_inserter( possibilities ) );

			AltList                func_candidates;
			std::vector< AltList > args_candidates;

			// For every possible function :
			// 	try matching the arguments to the parameters
			// 	not the other way around because we have more arguments than parameters
			SemanticError errors;
			for ( Alternative & func : funcFinder.get_alternatives() ) {
				try {
					PointerType * pointer = dynamic_cast< PointerType* >( func.expr->get_result()->stripReferences() );
					if( !pointer ) {
						throw SemanticError( "candidate not viable: not a pointer type\n", func.expr->get_result() );
					}

					FunctionType * function = dynamic_cast< FunctionType* >( pointer->get_base() );
					if( !function ) {
						throw SemanticError( "candidate not viable: not a function type\n", pointer->get_base() );
					}


					{
						auto param     = function->parameters.begin();
						auto param_end = function->parameters.end();

						if( !advance_to_mutex( param, param_end ) ) {
							throw SemanticError("candidate function not viable: no mutex parameters\n", function);
						}
					}

					Alternative newFunc( func );
					// Strip reference from function
					referenceToRvalueConversion( newFunc.expr );

					// For all the set of arguments we have try to match it with the parameter of the current function alternative
					for ( auto & argsList : possibilities ) {

						try {
							// Declare data structures need for resolution
							OpenVarSet openVars;
							AssertionSet resultNeed, resultHave;
							TypeEnvironment resultEnv;

							// Load type variables from arguemnts into one shared space
							simpleCombineEnvironments( argsList.begin(), argsList.end(), resultEnv );

							// Make sure we don't widen any existing bindings
							for ( auto & i : resultEnv ) {
								i.allowWidening = false;
							}

							// Find any unbound type variables
							resultEnv.extractOpenVars( openVars );

							auto param     = function->parameters.begin();
							auto param_end = function->parameters.end();

							// For every arguments of its set, check if it matches one of the parameter
							// The order is important
							for( auto & arg : argsList ) {

								// Ignore non-mutex arguments
								if( !advance_to_mutex( param, param_end ) ) {
									// We ran out of parameters but still have arguments
									// this function doesn't match
									throw SemanticError("candidate function not viable: too many mutex arguments\n", function);
								}

								// Check if the argument matches the parameter type in the current scope
								if( ! unify( (*param)->get_type(), arg.expr->get_result(), resultEnv, resultNeed, resultHave, openVars, this->indexer ) ) {
									// Type doesn't match
									stringstream ss;
									ss << "candidate function not viable: no known convertion from '";
									arg.expr->get_result()->print( ss );
									ss << "' to '";
									(*param)->get_type()->print( ss );
									ss << "'\n";
									throw SemanticError(ss.str(), function);
								}

								param++;
							}

							// All arguments match !

							// Check if parameters are missing
							if( advance_to_mutex( param, param_end ) ) {
								// We ran out of arguments but still have parameters left
								// this function doesn't match
								throw SemanticError("candidate function not viable: too few mutex arguments\n", function);
							}

							// All parameters match !

							// Finish the expressions to tie in the proper environments
							finishExpr( newFunc.expr, resultEnv );
							for( Alternative & alt : argsList ) {
								finishExpr( alt.expr, resultEnv );
							}

							// This is a match store it and save it for later
							func_candidates.push_back( newFunc );
							args_candidates.push_back( argsList );

						}
						catch( SemanticError &e ) {
							errors.append( e );
						}
					}
				}
				catch( SemanticError &e ) {
					errors.append( e );
				}
			}

			// Make sure we got the right number of arguments
			if( func_candidates.empty() )    { SemanticError top( "No alternatives for function in call to waitfor"  ); top.append( errors ); throw top; }
			if( args_candidates.empty() )    { SemanticError top( "No alternatives for arguments in call to waitfor" ); top.append( errors ); throw top; }
			if( func_candidates.size() > 1 ) { SemanticError top( "Ambiguous function in call to waitfor"            ); top.append( errors ); throw top; }
			if( args_candidates.size() > 1 ) { SemanticError top( "Ambiguous arguments in call to waitfor"           ); top.append( errors ); throw top; }


			// Swap the results from the alternative with the unresolved values.
			// Alternatives will handle deletion on destruction
			std::swap( clause.target.function, func_candidates.front().expr );
			for( auto arg_pair : group_iterate( clause.target.arguments, args_candidates.front() ) ) {
				std::swap ( std::get<0>( arg_pair), std::get<1>( arg_pair).expr );
			}

			// Resolve the conditions as if it were an IfStmt
			// Resolve the statments normally
			findSingleExpression( clause.condition, this->indexer );
			clause.statement->accept( *visitor );
		}


		if( stmt->timeout.statement ) {
			// Resolve the timeout as an size_t for now
			// Resolve the conditions as if it were an IfStmt
			// Resolve the statments normally
			findSingleExpression( stmt->timeout.time, new BasicType( noQualifiers, BasicType::LongLongUnsignedInt ), this->indexer );
			findSingleExpression( stmt->timeout.condition, this->indexer );
			stmt->timeout.statement->accept( *visitor );
		}

		if( stmt->orelse.statement ) {
			// Resolve the conditions as if it were an IfStmt
			// Resolve the statments normally
			findSingleExpression( stmt->orelse.condition, this->indexer );
			stmt->orelse.statement->accept( *visitor );
		}
	}

	template< typename T >
	bool isCharType( T t ) {
		if ( BasicType * bt = dynamic_cast< BasicType * >( t ) ) {
			return bt->get_kind() == BasicType::Char || bt->get_kind() == BasicType::SignedChar ||
				bt->get_kind() == BasicType::UnsignedChar;
		}
		return false;
	}

	void Resolver::previsit( SingleInit *singleInit ) {
		visit_children = false;
		// resolve initialization using the possibilities as determined by the currentObject cursor
		Expression * newExpr = new UntypedInitExpr( singleInit->get_value(), currentObject.getOptions() );
		findSingleExpression( newExpr, indexer );
		InitExpr * initExpr = strict_dynamic_cast< InitExpr * >( newExpr );

		// move cursor to the object that is actually initialized
		currentObject.setNext( initExpr->get_designation() );

		// discard InitExpr wrapper and retain relevant pieces
		newExpr = initExpr->expr;
		initExpr->expr = nullptr;
		std::swap( initExpr->env, newExpr->env );
		delete initExpr;

		// get the actual object's type (may not exactly match what comes back from the resolver due to conversions)
		Type * initContext = currentObject.getCurrentType();

		// check if actual object's type is char[]
		if ( ArrayType * at = dynamic_cast< ArrayType * >( initContext ) ) {
			if ( isCharType( at->get_base() ) ) {
				// check if the resolved type is char *
				if ( PointerType * pt = dynamic_cast< PointerType *>( newExpr->get_result() ) ) {
					if ( isCharType( pt->get_base() ) ) {
						// strip cast if we're initializing a char[] with a char *, e.g.  char x[] = "hello";
						CastExpr *ce = strict_dynamic_cast< CastExpr * >( newExpr );
						newExpr = ce->get_arg();
						ce->set_arg( nullptr );
						delete ce;
					}
				}
			}
		}

		// set initializer expr to resolved express
		singleInit->set_value( newExpr );

		// move cursor to next object in preparation for next initializer
		currentObject.increment();
	}

	void Resolver::previsit( ListInit * listInit ) {
		visit_children = false;
		// move cursor into brace-enclosed initializer-list
		currentObject.enterListInit();
		// xxx - fix this so that the list isn't copied, iterator should be used to change current element
		std::list<Designation *> newDesignations;
		for ( auto p : group_iterate(listInit->get_designations(), listInit->get_initializers()) ) {
			// iterate designations and initializers in pairs, moving the cursor to the current designated object and resolving
			// the initializer against that object.
			Designation * des = std::get<0>(p);
			Initializer * init = std::get<1>(p);
			newDesignations.push_back( currentObject.findNext( des ) );
			init->accept( *visitor );
		}
		// set the set of 'resolved' designations and leave the brace-enclosed initializer-list
		listInit->get_designations() = newDesignations; // xxx - memory management
		currentObject.exitListInit();

		// xxx - this part has not be folded into CurrentObject yet
		// } else if ( TypeInstType * tt = dynamic_cast< TypeInstType * >( initContext ) ) {
		// 	Type * base = tt->get_baseType()->get_base();
		// 	if ( base ) {
		// 		// know the implementation type, so try using that as the initContext
		// 		ObjectDecl tmpObj( "", Type::StorageClasses(), LinkageSpec::Cforall, nullptr, base->clone(), nullptr );
		// 		currentObject = &tmpObj;
		// 		visit( listInit );
		// 	} else {
		// 		// missing implementation type -- might be an unknown type variable, so try proceeding with the current init context
		// 		Parent::visit( listInit );
		// 	}
		// } else {
	}

	// ConstructorInit - fall back on C-style initializer
	void Resolver::fallbackInit( ConstructorInit * ctorInit ) {
		// could not find valid constructor, or found an intrinsic constructor
		// fall back on C-style initializer
		delete ctorInit->get_ctor();
		ctorInit->set_ctor( NULL );
		delete ctorInit->get_dtor();
		ctorInit->set_dtor( NULL );
		maybeAccept( ctorInit->get_init(), *visitor );
	}

	// needs to be callable from outside the resolver, so this is a standalone function
	void resolveCtorInit( ConstructorInit * ctorInit, const SymTab::Indexer & indexer ) {
		assert( ctorInit );
		PassVisitor<Resolver> resolver( indexer );
		ctorInit->accept( resolver );
	}

	void resolveStmtExpr( StmtExpr * stmtExpr, const SymTab::Indexer & indexer ) {
		assert( stmtExpr );
		PassVisitor<Resolver> resolver( indexer );
		stmtExpr->accept( resolver );
	}

	void Resolver::previsit( ConstructorInit *ctorInit ) {
		visit_children = false;
		// xxx - fallback init has been removed => remove fallbackInit function and remove complexity from FixInit and remove C-init from ConstructorInit
		maybeAccept( ctorInit->get_ctor(), *visitor );
		maybeAccept( ctorInit->get_dtor(), *visitor );

		// found a constructor - can get rid of C-style initializer
		delete ctorInit->get_init();
		ctorInit->set_init( NULL );

		// intrinsic single parameter constructors and destructors do nothing. Since this was
		// implicitly generated, there's no way for it to have side effects, so get rid of it
		// to clean up generated code.
		if ( InitTweak::isIntrinsicSingleArgCallStmt( ctorInit->get_ctor() ) ) {
			delete ctorInit->get_ctor();
			ctorInit->set_ctor( NULL );
		}

		if ( InitTweak::isIntrinsicSingleArgCallStmt( ctorInit->get_dtor() ) ) {
			delete ctorInit->get_dtor();
			ctorInit->set_dtor( NULL );
		}

		// xxx - todo -- what about arrays?
		// if ( dtor == NULL && InitTweak::isIntrinsicCallStmt( ctorInit->get_ctor() ) ) {
		// 	// can reduce the constructor down to a SingleInit using the
		// 	// second argument from the ctor call, since
		// 	delete ctorInit->get_ctor();
		// 	ctorInit->set_ctor( NULL );

		// 	Expression * arg =
		// 	ctorInit->set_init( new SingleInit( arg ) );
		// }
	}
} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
