#include "collector.h"

#include <stdint.h>
#include <stdlib.h>
#include <string.h>

#include "fstream.h"

#include "gc_tools.h"
#include "state.h"
// #include "memory_pool.h"

void* gc_finish_alloc_block(void* block, size_t actual_size, size_t target_size);
void gc_assign_reference(void** ref, gc_object_header* ptr);

gcpointer_t** gc_find_previous_ref(gcpointer_t* target)
{
	if(!(target)) return NULL;

	bool managed = gc_is_managed(target);
	gc_object_header* obj = gc_get_object_ptr((void*)target->ptr);

	check(gc_is_valide(obj));

	gcpointer_t** prev_next_ptr = managed ? &obj->type_chain : &obj->root_chain;
	while((*prev_next_ptr) && (*prev_next_ptr) != target)
	{
		prev_next_ptr = &(*prev_next_ptr)->next;
	}

	return prev_next_ptr;
}

void* gc_allocate(size_t target_size)
{
	size_t size = gc_compute_size(target_size + sizeof(gc_object_header));

	check(size < POOL_SIZE_BYTES);

	void* block = NULL;
	gc_state* gc = gc_get_state();

	if((intptr_t)(block = gc_try_allocate(gc, size))) return gc_finish_alloc_block(block, size, target_size);

	gc_collect(gc);

	if((intptr_t)(block = gc_try_allocate(gc, size))) return gc_finish_alloc_block(block, size, target_size);

	gc_allocate_pool(gc);

	if((intptr_t)(block = gc_try_allocate(gc, size))) return gc_finish_alloc_block(block, size, target_size);

	ofstream_stderr() | "ERROR: allocation in new pool failed" | endl;
	abort();

	return NULL;
}

void* gc_finish_alloc_block(void* block, size_t actual_size, size_t target_size)
{
	void* data = (void*)(((intptr_t)block) + sizeof(gc_object_header));
	void* header = block;

	check(((intptr_t)data) > ((intptr_t)block));
	check(((intptr_t)data) >= ((intptr_t)header));
	check(is_aligned(data));
	check(((intptr_t)data) + target_size <= ((intptr_t)block) + actual_size);

	gc_object_header* obj = gc_object_header_placement_ctor(header, actual_size);

	(void)obj; //remove unsused warning since this is for debug
	check(obj == get_object_ptr(data));

	gc_register_allocation(gc_get_state(), actual_size);

	return data;
}

// 	void process_reference(void** ref, std::vector<void**>& worklist)
// 	{
// 		check(!gc::gc_get_state().is_in_heap(ref));
//
// 		if(gc_object_header* ptr = get_object_ptr(*ref))
// 		{
// 			if(!ptr->is_forwarded)
// 			{
// 				copy_object(ptr);
//
// 				scan_object(ptr->forward, worklist);
//
// 				assign_reference(ref, ptr->forward);
// 			}
// 			else
// 			{
// 				//duplication to help debug
// 				assign_reference(ref, ptr->forward);
// 			}
// 		}
// 	}
//
// 	void assign_reference(void** ref, gc_object_header* ptr)
// 	{
// 		void* address = (void*)(((intptr_t)ptr) + sizeof(gc_object_header));
//
// 		write_aligned_ptr(ref, address);
// 	}
//
// 	gc_object_header* copy_object(gc_object_header* ptr)
// 	{
// 		check(!ptr->forward);
// 		check(!ptr->is_forwarded);
// 		check(pool_of(ptr)->is_from_space());
//
// 		memory_pool* pool = pool_of(ptr)->mirror();
//
// 		void* new_block = pool->allocate(ptr->size, false);
//
// 		std::memcpy(new_block, ptr, ptr->size);
//
// 		gc_object_header* fwd_ptr = new (new_block) gc_object_header(ptr);
//
// 		ptr->forward = fwd_ptr;
// 		ptr->is_forwarded = true;
//
// 		return fwd_ptr;
// 	}
//
// 	void scan_object(gc_object_header* object, std::vector<void**>& worklist)
// 	{
// 		gcpointer_base* type = object->type_chain;
// 		while(type)
// 		{
// 			check(((intptr_t)type) > ((intptr_t)object));
// 			check(((intptr_t)type) < ((intptr_t)((intptr_t)object) + object->size));
//
// 			check(gc::gc_get_state().is_in_to_space(&type->m_ptr));
//
// 			worklist.push_back(&type->m_ptr);
//
// 			type = type->m_next;
// 		}
// 	}
// };
