//
// Cforall Version 1.0.0 Copyright (C) 2018 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Demangle.cc -- Convert a mangled name into a human readable name.
//
// Author           : Rob Schluntz
// Created On       : Thu Jul 19 12:52:41 2018
// Last Modified By : Andrew Beach
// Last Modified On : Mon Nov  6 15:59:00 2023
// Update Count     : 12
//

#include <algorithm>
#include <sstream>

#include "CodeGen/GenType.h"
#include "CodeGen/OperatorTable.h"
#include "Common/PassVisitor.h"
#include "Common/utility.h"								// isPrefix
#include "Mangler.h"
#include "SynTree/Type.h"
#include "SynTree/Declaration.h"

#define DEBUG
#ifdef DEBUG
#define PRINT(x) x
#else
#define PRINT(x) {}
#endif

namespace SymTab {
	namespace Mangler {
		namespace {
			struct StringView {
			private:
				std::string str;
				size_t idx = 0;
				// typedef Type * (StringView::*parser)(Type::Qualifiers);
				typedef std::function<Type * (Type::Qualifiers)> parser;
				std::vector<std::pair<std::string, parser>> parsers;
			public:
				StringView(const std::string & str);

				bool done() const { return idx >= str.size(); }
				char cur() const { assert(! done()); return str[idx]; }

				bool expect(char ch) { return str[idx++] == ch;	}
				void next(size_t inc = 1) { idx += inc; }

				/// determines if `pref` is a prefix of `str`
				bool isPrefix(const std::string & pref);
				bool extractNumber(size_t & out);
				bool extractName(std::string & out);
				bool stripMangleName(std::string & name);

				Type * parseFunction(Type::Qualifiers tq);
				Type * parseTuple(Type::Qualifiers tq);
				Type * parseVoid(Type::Qualifiers tq);
				Type * parsePointer(Type::Qualifiers tq);
				Type * parseArray(Type::Qualifiers tq);
				Type * parseStruct(Type::Qualifiers tq);
				Type * parseUnion(Type::Qualifiers tq);
				Type * parseEnum(Type::Qualifiers tq);
				Type * parseType(Type::Qualifiers tq);

				Type * parseType();
				bool parse(std::string & name, Type *& type);
			};

			StringView::StringView(const std::string & str) : str(str) {
				// basic types
				for (size_t k = 0; k < BasicType::NUMBER_OF_BASIC_TYPES; ++k) {
					parsers.emplace_back(Encoding::basicTypes[k], [k](Type::Qualifiers tq) {
						PRINT( std::cerr << "basic type: " << k << std::endl; )
						return new BasicType(tq, (BasicType::Kind)k);
					});
				}
				// type variable types
				for (size_t k = 0; k < TypeDecl::NUMBER_OF_KINDS; ++k) {
					static const std::string typeVariableNames[] = { "DT", "DST", "OT", "FT", "TT", "ALT", };
					static_assert(
						sizeof(typeVariableNames)/sizeof(typeVariableNames[0]) == TypeDecl::NUMBER_OF_KINDS,
						"Each type variable kind should have a demangle name prefix"
					);
					parsers.emplace_back(Encoding::typeVariables[k], [k, this](Type::Qualifiers tq) -> TypeInstType * {
						PRINT( std::cerr << "type variable type: " << k << std::endl; )
						size_t N;
						if (! extractNumber(N)) return nullptr;
						return new TypeInstType(tq, toString(typeVariableNames[k], N), (TypeDecl::Kind)k != TypeDecl::Ftype);
					});
				}
				// everything else
				parsers.emplace_back(Encoding::void_t, [this](Type::Qualifiers tq) { return parseVoid(tq); });
				parsers.emplace_back(Encoding::function, [this](Type::Qualifiers tq) { return parseFunction(tq); });
				parsers.emplace_back(Encoding::pointer, [this](Type::Qualifiers tq) { return parsePointer(tq); });
				parsers.emplace_back(Encoding::array, [this](Type::Qualifiers tq) { return parseArray(tq); });
				parsers.emplace_back(Encoding::tuple, [this](Type::Qualifiers tq) { return parseTuple(tq); });
				parsers.emplace_back(Encoding::struct_t, [this](Type::Qualifiers tq) { return parseStruct(tq); });
				parsers.emplace_back(Encoding::union_t, [this](Type::Qualifiers tq) { return parseUnion(tq); });
				parsers.emplace_back(Encoding::enum_t, [this](Type::Qualifiers tq) { return parseEnum(tq); });
				parsers.emplace_back(Encoding::type, [this](Type::Qualifiers tq) { return parseType(tq); });
				parsers.emplace_back(Encoding::zero, [](Type::Qualifiers tq) { return new ZeroType(tq); });
				parsers.emplace_back(Encoding::one, [](Type::Qualifiers tq) { return new OneType(tq); });
			}

			bool StringView::extractNumber(size_t & out) {
				std::stringstream numss;
				if (idx >= str.size()) return false;
				while (isdigit(str[idx])) {
					numss << str[idx];
					++idx;
					if (idx == str.size()) break;
				}
				if (! (numss >> out)) return false;
				PRINT( std::cerr << "extractNumber success: " << out << std::endl; )
				return true;
			}

			bool StringView::extractName(std::string & out) {
				size_t len;
				if (! extractNumber(len)) return false;
				if (idx+len > str.size()) return false;
				out = str.substr(idx, len);
				idx += len;
				PRINT( std::cerr << "extractName success: " << out << std::endl; )
				return true;
			}

			bool StringView::isPrefix(const std::string & pref) {
				// if ( pref.size() > str.size()-idx ) return false;
				// auto its = std::mismatch( pref.begin(), pref.end(), std::next(str.begin(), idx) );
				// if (its.first == pref.end()) {
				// 	idx += pref.size();
				// 	return true;
				// }

				// This update is untested because there are no tests for this code.
				if ( ::isPrefix( str, pref, idx ) ) {
					idx += pref.size();
					return true;
				}
				return false;
			}

			// strips __NAME__cfa__TYPE_N, where N is [0-9]+: returns str is a match is found, returns empty string otherwise
			bool StringView::stripMangleName(std::string & name) {
				PRINT( std::cerr << "====== " << str.size() << " " << str << std::endl; )
				if (str.size() < 2+Encoding::manglePrefix.size()) return false; // +2 for at least _1 suffix
				if ( ! isPrefix(Encoding::manglePrefix) || ! isdigit(str.back() ) ) return false;

				// get name
				if (! extractName(name)) return false;

				// find bounds for type
				PRINT( std::cerr << idx << " " << str.size() << std::endl; )
				PRINT( std::cerr << "[");
				while (isdigit(str.back())) {
					PRINT(std::cerr << ".");
					str.pop_back();
					if (str.size() <= idx) return false;
				}
				PRINT( std::cerr << "]" << std::endl );
				if (str.back() != '_') return false;
				str.pop_back();
				PRINT( std::cerr << str.size() << " " << name << " " << str.substr(idx) << std::endl; )
				return str.size() > idx;
			}

			Type * StringView::parseFunction(Type::Qualifiers tq) {
				PRINT( std::cerr << "function..." << std::endl; )
				if (done()) return nullptr;
				FunctionType * ftype = new FunctionType( tq, false );
				std::unique_ptr<Type> manager(ftype);
				Type * retVal = parseType();
				if (! retVal) return nullptr;
				PRINT( std::cerr << "with return type: " << retVal << std::endl; )
				ftype->returnVals.push_back(ObjectDecl::newObject("", retVal, nullptr));
				if (done() || ! expect('_')) return nullptr;
				while (! done()) {
					PRINT( std::cerr << "got ch: " << cur() << std::endl; )
					if (cur() == '_') return manager.release();
					Type * param = parseType();
					if (! param) return nullptr;
					PRINT( std::cerr << "with parameter : " << param << std::endl; )
					ftype->parameters.push_back(ObjectDecl::newObject("", param, nullptr));
				}
				return nullptr;
			}

			Type * StringView::parseTuple(Type::Qualifiers tq) {
				PRINT( std::cerr << "tuple..." << std::endl; )
				std::list< Type * > types;
				size_t ncomponents;
				if (! extractNumber(ncomponents)) return nullptr;
				for (size_t i = 0; i < ncomponents; ++i) {
					// TODO: delete all on return
					if (done()) return nullptr;
					PRINT( std::cerr << "got ch: " << cur() << std::endl; )
					Type * t = parseType();
					if (! t) return nullptr;
					PRINT( std::cerr << "with type : " << t << std::endl; )
					types.push_back(t);
				}
				return new TupleType( tq, types );
			}

			Type * StringView::parseVoid(Type::Qualifiers tq) {
				return new VoidType( tq );
			}

			Type * StringView::parsePointer(Type::Qualifiers tq) {
				PRINT( std::cerr << "pointer..." << std::endl; )
				Type * t = parseType();
				if (! t) return nullptr;
				return new PointerType( tq, t );
			}

			Type * StringView::parseArray(Type::Qualifiers tq) {
				PRINT( std::cerr << "array..." << std::endl; )
				size_t length;
				if (! extractNumber(length)) return nullptr;
				Type * t = parseType();
				if (! t) return nullptr;
				return new ArrayType( tq, t, new ConstantExpr( Constant::from_ulong(length) ), false, false );
			}

			Type * StringView::parseStruct(Type::Qualifiers tq) {
				PRINT( std::cerr << "struct..." << std::endl; )
				std::string name;
				if (! extractName(name)) return nullptr;
				return new StructInstType(tq, name);
			}

			Type * StringView::parseUnion(Type::Qualifiers tq) {
				PRINT( std::cerr << "union..." << std::endl; )
				std::string name;
				if (! extractName(name)) return nullptr;
				return new UnionInstType(tq, name);
			}

			Type * StringView::parseEnum(Type::Qualifiers tq) {
				PRINT( std::cerr << "enum..." << std::endl; )
				std::string name;
				if (! extractName(name)) return nullptr;
				return new EnumInstType(tq, name);
			}

			Type * StringView::parseType(Type::Qualifiers tq) {
				PRINT( std::cerr << "type..." << std::endl; )
				std::string name;
				if (! extractName(name)) return nullptr;
				PRINT( std::cerr << "typename..." << name << std::endl; )
				return new TypeInstType(tq, name, false);
			}

			Type * StringView::parseType() {
				if (done()) return nullptr;

				std::list<TypeDecl *> forall;
				if (isPrefix(Encoding::forall)) {
					PRINT( std::cerr << "polymorphic with..." << std::endl; )
					size_t dcount, fcount, vcount, acount;
					if (! extractNumber(dcount)) return nullptr;
					PRINT( std::cerr << dcount << " dtypes" << std::endl; )
					if (! expect('_')) return nullptr;
					if (! extractNumber(fcount)) return nullptr;
					PRINT( std::cerr << fcount << " ftypes" << std::endl; )
					if (! expect('_')) return nullptr;
					if (! extractNumber(vcount)) return nullptr;
					PRINT( std::cerr << vcount << " ttypes" << std::endl; )
					if (! expect('_')) return nullptr;
					if (! extractNumber(acount)) return nullptr;
					PRINT( std::cerr << acount << " assertions" << std::endl; )
					if (! expect('_')) return nullptr;
					for (size_t i = 0; i < acount; ++i) {
						// TODO: need to recursively parse assertions, but for now just return nullptr so that
						// demangler does not crash if there are assertions
						return nullptr;
					}
					if (! expect('_')) return nullptr;
				}

				// qualifiers
				Type::Qualifiers tq;
				while (true) {
					auto qual = std::find_if(Encoding::qualifiers.begin(), Encoding::qualifiers.end(), [this](decltype(Encoding::qualifiers)::value_type val) {
						return isPrefix(val.second);
					});
					if (qual == Encoding::qualifiers.end()) break;
					tq |= qual->first;
				}

				// find the correct type parser and use it
				auto iter = std::find_if(parsers.begin(), parsers.end(), [this](std::pair<std::string, parser> & p) {
					return isPrefix(p.first);
				});
				assertf(iter != parsers.end(), "Unhandled type letter: %c at index: %zd", cur(), idx);
				Type * ret = iter->second(tq);
				if (! ret) return nullptr;
				ret->forall = std::move(forall);
				return ret;
			}

			bool StringView::parse(std::string & name, Type *& type) {
				if (! stripMangleName(name)) return false;
				PRINT( std::cerr << "stripped name: " << name << std::endl; )
				Type * t = parseType();
				if (! t) return false;
				type = t;
				return true;
			}

			std::string demangle(const std::string & mangleName) {
				SymTab::Mangler::StringView view(mangleName);
				std::string name;
				Type * type = nullptr;
				if (! view.parse(name, type)) return mangleName;
				auto info = CodeGen::operatorLookupByOutput(name);
				if (info) name = info->inputName;
				std::unique_ptr<Type> manager(type);
				return CodeGen::genType(type, name);
			}
		} // namespace
	} // namespace Mangler
} // namespace SymTab

extern "C" {
	char * cforall_demangle(const char * mangleName, int option __attribute__((unused))) {
		const std::string & demangleName = SymTab::Mangler::demangle(mangleName);
		return strdup(demangleName.c_str());
	}
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
