//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// ExpressionNode.cc --
//
// Author           : Rodolfo G. Esteves
// Created On       : Sat May 16 13:17:07 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Sun Aug  7 09:23:12 2016
// Update Count     : 437
//

#include <cassert>
#include <cctype>
#include <climits>
#include <cstdio>
#include <algorithm>
#include <sstream>

#include "ParseNode.h"
#include "TypeData.h"
#include "SynTree/Constant.h"
#include "SynTree/Expression.h"
#include "SynTree/Declaration.h"
#include "Common/UnimplementedError.h"
#include "parseutility.h"
#include "Common/utility.h"

using namespace std;

ExpressionNode::ExpressionNode() : ParseNode() {}

ExpressionNode::ExpressionNode( const string *name ) : ParseNode( name ) {}

ExpressionNode::ExpressionNode( const ExpressionNode &other ) : ParseNode( other.name ), extension( other.extension ) {
	if ( other.argName ) {
		std::cout << "ExpressionNode" << std::endl;
		argName = other.argName->clone();
	} else {
		argName = 0;
	} // if
}

ExpressionNode * ExpressionNode::set_argName( const std::string *aName ) {
	argName = new VarRefNode( aName );
	return this;
}

ExpressionNode * ExpressionNode::set_argName( ExpressionNode *aDesignator ) {
	argName = aDesignator;
	return this;
}

void ExpressionNode::printDesignation( std::ostream &os, int indent ) const {
	if ( argName ) {
		os << string( indent, ' ' ) << "(designated by:  ";
		argName->printOneLine( os, indent );
		os << ")" << std::endl;
	} // if
}

//##############################################################################

NullExprNode::NullExprNode() {}

NullExprNode *NullExprNode::clone() const {
	return new NullExprNode();
}

void NullExprNode::print( std::ostream & os, int indent ) const {
	printDesignation( os );
	os << "null expression";
}

void NullExprNode::printOneLine( std::ostream & os, int indent ) const {
	printDesignation( os );
	os << "null";
}

Expression *NullExprNode::build() const {
	return 0;
}

//##############################################################################

// Difficult to separate extra parts of constants during lexing because actions are not allow in the middle of patterns:
//
//		prefix action constant action suffix
//
// Alternatively, breaking a pattern using BEGIN does not work if the following pattern can be empty:
//
//		constant BEGIN CONT ...
//		<CONT>(...)? BEGIN 0 ... // possible empty suffix
//
// because the CONT rule is NOT triggered if the pattern is empty. Hence, constants are reparsed here to determine their
// type.

static Type::Qualifiers emptyQualifiers;				// no qualifiers on constants

static inline bool checkU( char c ) { return c == 'u' || c == 'U'; }
static inline bool checkL( char c ) { return c == 'l' || c == 'L'; }
static inline bool checkF( char c ) { return c == 'f' || c == 'F'; }
static inline bool checkD( char c ) { return c == 'd' || c == 'D'; }
static inline bool checkI( char c ) { return c == 'i' || c == 'I'; }
static inline bool checkX( char c ) { return c == 'x' || c == 'X'; }

ConstantNode *build_constantInteger( std::string & str ) {
	static const BasicType::Kind kind[2][3] = {
		{ BasicType::SignedInt, BasicType::LongSignedInt, BasicType::LongLongSignedInt },
		{ BasicType::UnsignedInt, BasicType::LongUnsignedInt, BasicType::LongLongUnsignedInt },
	};
	bool dec = true, Unsigned = false;					// decimal, unsigned constant
	int size;											// 0 => int, 1 => long, 2 => long long
	unsigned long long v;								// converted integral value
	size_t last = str.length() - 1;						// last character of constant

	if ( str[0] == '0' ) {								// octal/hex constant ?
		dec = false;
		if ( last != 0 && checkX( str[1] ) ) {			// hex constant ?
			sscanf( (char *)str.c_str(), "%llx", &v );
			//printf( "%llx %llu\n", v, v );
		} else {										// octal constant
			sscanf( (char *)str.c_str(), "%llo", &v );
			//printf( "%llo %llu\n", v, v );
		} // if
	} else {											// decimal constant ?
		sscanf( (char *)str.c_str(), "%llu", &v );
		//printf( "%llu %llu\n", v, v );
	} // if

	if ( v <= INT_MAX ) {								// signed int
		size = 0;
	} else if ( v <= UINT_MAX && ! dec ) {				// unsigned int
		size = 0;
		Unsigned = true;								// unsigned
	} else if ( v <= LONG_MAX ) {						// signed long int
		size = 1;
	} else if ( v <= ULONG_MAX && ( ! dec || LONG_MAX == LLONG_MAX ) ) { // signed long int
		size = 1;
		Unsigned = true;								// unsigned long int
	} else if ( v <= LLONG_MAX ) {						// signed long long int
		size = 2;
	} else {											// unsigned long long int
		size = 2;
		Unsigned = true;								// unsigned long long int
	} // if

	if ( checkU( str[last] ) ) {						// suffix 'u' ?
		Unsigned = true;
		if ( last > 0 && checkL( str[last - 1] ) ) {	// suffix 'l' ?
			size = 1;
			if ( last > 1 && checkL( str[last - 2] ) ) { // suffix 'll' ?
				size = 2;
			} // if
		} // if
	} else if ( checkL( str[ last ] ) ) {				// suffix 'l' ?
		size = 1;
		if ( last > 0 && checkL( str[last - 1] ) ) {	// suffix 'll' ?
			size = 2;
			if ( last > 1 && checkU( str[last - 2] ) ) { // suffix 'u' ?
				Unsigned = true;
			} // if
		} else {
			if ( last > 0 && checkU( str[last - 1] ) ) { // suffix 'u' ?
				Unsigned = true;
			} // if
		} // if
	} // if

	return new ConstantNode( new ConstantExpr( Constant( new BasicType( emptyQualifiers, kind[Unsigned][size] ), str ) ) );
} // build_constantInteger

ConstantNode *build_constantFloat( std::string & str ) {
	static const BasicType::Kind kind[2][3] = {
		{ BasicType::Float, BasicType::Double, BasicType::LongDouble },
		{ BasicType::FloatComplex, BasicType::DoubleComplex, BasicType::LongDoubleComplex },
	};

	bool complx = false;								// real, complex
	int size = 1;										// 0 => float, 1 => double (default), 2 => long double
	// floating-point constant has minimum of 2 characters: 1. or .1
	size_t last = str.length() - 1;

	if ( checkI( str[last] ) ) {						// imaginary ?
		complx = true;
		last -= 1;										// backup one character
	} // if

	if ( checkF( str[last] ) ) {						// float ?
		size = 0;
	} else if ( checkD( str[last] ) ) {					// double ?
		size = 1;
	} else if ( checkL( str[last] ) ) {					// long double ?
		size = 2;
	} // if
	if ( ! complx && checkI( str[last - 1] ) ) {		// imaginary ?
		complx = true;
	} // if

	return new ConstantNode( new ConstantExpr( Constant( new BasicType( emptyQualifiers, kind[complx][size] ), str ) ) );
} // build_constantFloat

ConstantNode *build_constantChar( std::string & str ) {
	return new ConstantNode( new ConstantExpr( Constant( new BasicType( emptyQualifiers, BasicType::Char ), str ) ) );
} // build_constantChar

ConstantNode *build_constantStr( std::string & str ) {
	// string should probably be a primitive type
	ArrayType *at = new ArrayType( emptyQualifiers, new BasicType( emptyQualifiers, BasicType::Char ),
				new ConstantExpr( Constant( new BasicType( emptyQualifiers, BasicType::UnsignedInt ),
											toString( str.size()+1-2 ) ) ),  // +1 for '\0' and -2 for '"'
								   false, false );
	return new ConstantNode( new ConstantExpr( Constant( at, str ) ) );
} // build_constantStr

//##############################################################################

//Expression *build_varref( ExpressionNode expr ) {
//	return new NameExpr( get_name(), maybeBuild<Expression>( get_argName() ) );
//}

VarRefNode::VarRefNode( const string *name, bool labelp ) : ExpressionNode( name ), isLabel( labelp ) {}

VarRefNode::VarRefNode( const VarRefNode &other ) : ExpressionNode( other ), isLabel( other.isLabel ) {
}

Expression *VarRefNode::build() const {
	return new NameExpr( get_name(), maybeBuild< Expression >( get_argName() ) );
}

void VarRefNode::printOneLine( std::ostream &os, int indent ) const {
	printDesignation( os );
	os << get_name() << ' ';
}

void VarRefNode::print( std::ostream &os, int indent ) const {
	printDesignation( os );
	os << string( indent, ' ' ) << "Referencing: ";
	os << "Variable: " << get_name();
	os << endl;
}

//##############################################################################

DesignatorNode::DesignatorNode( ExpressionNode *expr, bool isArrayIndex ) : isArrayIndex( isArrayIndex ) {
	set_argName( expr );
	assert( get_argName() );

	if ( ! isArrayIndex ) {
		if ( VarRefNode * var = dynamic_cast< VarRefNode * >( expr ) ) {

			stringstream ss( var->get_name() );
			double value;
			if ( ss >> value ) {
				// this is a floating point constant. It MUST be ".0" or ".1", otherwise the program is invalid
				if ( ! (var->get_name() == ".0" || var->get_name() == ".1") ) {
					throw SemanticError( "invalid designator name: " + var->get_name() );
				} // if
				var->set_name( var->get_name().substr(1) );
			} // if
		} // if
	} // if
}

DesignatorNode::DesignatorNode( const DesignatorNode &other ) : ExpressionNode( other ), isArrayIndex( other.isArrayIndex ) {
}

class DesignatorFixer : public Mutator {
public:
	virtual Expression* mutate( NameExpr *nameExpr ) {
		if ( nameExpr->get_name() == "0" || nameExpr->get_name() == "1" ) {
			Constant val( new BasicType( Type::Qualifiers(), BasicType::SignedInt ), nameExpr->get_name() );
			delete nameExpr;
			return new ConstantExpr( val );
		}
		return nameExpr;
	}
};

Expression *DesignatorNode::build() const {
	Expression * ret = maybeBuild<Expression>(get_argName());

	if ( isArrayIndex ) {
		// need to traverse entire structure and change any instances of 0 or 1 to ConstantExpr
		DesignatorFixer fixer;
		ret = ret->acceptMutator( fixer );
	} // if

	return ret;
}

void DesignatorNode::printOneLine( std::ostream &os, int indent ) const {
	if ( get_argName() ) {
		if ( isArrayIndex ) {
			os << "[";
			get_argName()->printOneLine( os, indent );
			os << "]";
		} else {
			os << ".";
			get_argName()->printOneLine( os, indent );
		}
	} // if
}

void DesignatorNode::print( std::ostream &os, int indent ) const {
	if ( get_argName() ) {
		if ( isArrayIndex ) {
			os << "[";
			get_argName()->print( os, indent );
			os << "]";
		} else {
			os << ".";
			get_argName()->print( os, indent );
		}
	} // if
}

//##############################################################################

static const char *OperName[] = {
	// diadic
	"SizeOf", "AlignOf", "OffsetOf", "?+?", "?-?", "?*?", "?/?", "?%?", "||", "&&",
	"?|?", "?&?", "?^?", "Cast", "?<<?", "?>>?", "?<?", "?>?", "?<=?", "?>=?", "?==?", "?!=?",
	"?=?", "?*=?", "?/=?", "?%=?", "?+=?", "?-=?", "?<<=?", "?>>=?", "?&=?", "?^=?", "?|=?",
	"?[?]", "...",
	// monadic
	"+?", "-?", "AddressOf", "*?", "!?", "~?", "++?", "?++", "--?", "?--", "&&"
};

//##############################################################################

Expression *build_cast( TypeValueNode * arg, ExpressionNode *expr_node ) {
	DeclarationNode *decl_node = arg->get_decl();

	Type *targetType = decl_node->buildType();
	if ( dynamic_cast< VoidType* >( targetType ) ) {
		delete targetType;
		return new CastExpr( maybeBuild<Expression>(expr_node) );
	} else {
		return new CastExpr( maybeBuild<Expression>(expr_node), targetType );
	} // if
}

Expression *build_fieldSel( ExpressionNode *expr_node, VarRefNode *member ) {
	NameExpr* memberExpr = dynamic_cast<NameExpr*> ( maybeBuild<Expression>( member) );
	assert( memberExpr );
	UntypedMemberExpr *ret = new UntypedMemberExpr( memberExpr->get_name(), maybeBuild<Expression>(expr_node) );
	delete member;
	return ret;
}

Expression *build_pfieldSel( ExpressionNode *expr_node, VarRefNode *member ) {
	NameExpr* memberExpr = dynamic_cast<NameExpr*> ( maybeBuild<Expression>( member) );
	assert( memberExpr );
	UntypedExpr *deref = new UntypedExpr( new NameExpr( "*?" ) );
	deref->get_args().push_back( maybeBuild<Expression>(expr_node) );
	UntypedMemberExpr *ret = new UntypedMemberExpr( memberExpr->get_name(), deref );
	delete member;
	return ret;
}

Expression *build_addressOf( ExpressionNode *expr_node ) {
		return new AddressExpr( maybeBuild<Expression>(expr_node) );
}
Expression *build_sizeOf( ExpressionNode *expr_node ) {
	if ( TypeValueNode * arg = dynamic_cast<TypeValueNode *>( expr_node ) ) {
		return new SizeofExpr( arg->get_decl()->buildType() );
	} else {
		return new SizeofExpr( maybeBuild<Expression>(expr_node) );
	} // if
}
Expression *build_alignOf( ExpressionNode *expr_node ) {
	if ( TypeValueNode * arg = dynamic_cast<TypeValueNode *>( expr_node ) ) {
		return new AlignofExpr( arg->get_decl()->buildType() );
	} else {
		return new AlignofExpr( maybeBuild<Expression>(expr_node) );
	} // if
}
Expression *build_offsetOf( TypeValueNode * arg, VarRefNode *member ) {
	NameExpr *memberExpr = dynamic_cast<NameExpr *>( maybeBuild<Expression>( member ) );
	assert( memberExpr );
	return new UntypedOffsetofExpr( arg->get_decl()->buildType(), memberExpr->get_name() );
}

Expression *build_and_or( ExpressionNode *expr_node1, ExpressionNode *expr_node2, bool kind ) {
	return new LogicalExpr( notZeroExpr( maybeBuild<Expression>(expr_node1) ), notZeroExpr( maybeBuild<Expression>(expr_node2) ), kind );
}

Expression *build_unary_val( OperKinds op, ExpressionNode *expr_node ) {
	std::list<Expression *> args;
	args.push_back( maybeBuild<Expression>(expr_node) );
	return new UntypedExpr( new NameExpr( OperName[ (int)op ] ), args );
}
Expression *build_unary_ptr( OperKinds op, ExpressionNode *expr_node ) {
	std::list<Expression *> args;
	args.push_back( new AddressExpr( maybeBuild<Expression>(expr_node) ) );
	return new UntypedExpr( new NameExpr( OperName[ (int)op ] ), args );
}
Expression *build_binary_val( OperKinds op, ExpressionNode *expr_node1, ExpressionNode *expr_node2 ) {
	std::list<Expression *> args;
	args.push_back( maybeBuild<Expression>(expr_node1) );
	args.push_back( maybeBuild<Expression>(expr_node2) );
	return new UntypedExpr( new NameExpr( OperName[ (int)op ] ), args );
}
Expression *build_binary_ptr( OperKinds op, ExpressionNode *expr_node1, ExpressionNode *expr_node2 ) {
	std::list<Expression *> args;
	args.push_back( new AddressExpr( maybeBuild<Expression>(expr_node1) ) );
	args.push_back( maybeBuild<Expression>(expr_node2) );
	return new UntypedExpr( new NameExpr( OperName[ (int)op ] ), args );
}

Expression *build_cond( ExpressionNode *expr_node1, ExpressionNode *expr_node2, ExpressionNode *expr_node3 ) {
	return new ConditionalExpr( notZeroExpr( maybeBuild<Expression>(expr_node1) ), maybeBuild<Expression>(expr_node2), maybeBuild<Expression>(expr_node3) );
}

Expression *build_comma( ExpressionNode *expr_node1, ExpressionNode *expr_node2 ) {
	return new CommaExpr( maybeBuild<Expression>(expr_node1), maybeBuild<Expression>(expr_node2) );
}

Expression *build_attr( VarRefNode *var, ExpressionNode * expr ) {
	if ( TypeValueNode * arg = dynamic_cast<TypeValueNode *>( expr ) ) {
		return new AttrExpr( maybeBuild<Expression>(var), arg->get_decl()->buildType() );
	} else {
		return new AttrExpr( maybeBuild<Expression>(var), maybeBuild<Expression>(expr) );
	} // if
}

Expression *build_tuple( ExpressionNode * expr ) {
	TupleExpr *ret = new TupleExpr();
	buildList( expr, ret->get_exprs() );
	return ret;
}

Expression *build_func( ExpressionNode * function, ExpressionNode * expr ) {
	std::list<Expression *> args;

	buildList( expr, args );
	return new UntypedExpr( maybeBuild<Expression>(function), args, nullptr );
}

Expression *build_range( ExpressionNode * low, ExpressionNode *high ) {
	Expression *low_cexpr = maybeBuild<Expression>( low );
	Expression *high_cexpr = maybeBuild<Expression>( high );
	return new RangeExpr( low_cexpr, high_cexpr );
}

//##############################################################################

Expression *AsmExprNode::build() const {
	return new AsmExpr( maybeBuild< Expression >( inout ), (ConstantExpr *)maybeBuild<Expression>(constraint), maybeBuild<Expression>(operand) );
}

void AsmExprNode::print( std::ostream &os, int indent ) const {
	os << string( indent, ' ' ) << "Assembler Expression:" << endl;
	if ( inout ) {
		os << string( indent, ' ' ) << "inout: " << std::endl;
		inout->print( os, indent + 2 );
	} // if
	if ( constraint ) {
		os << string( indent, ' ' ) << "constraint: " << std::endl;
		constraint->print( os, indent + 2 );
	} // if
	if ( operand ) {
		os << string( indent, ' ' ) << "operand: " << std::endl;
		operand->print( os, indent + 2 );
	} // if
}

void AsmExprNode::printOneLine( std::ostream &os, int indent ) const {
	printDesignation( os );
	os << "( ";
	if ( inout ) inout->printOneLine( os, indent + 2 );
	os << ", ";
	if ( constraint ) constraint->printOneLine( os, indent + 2 );
	os << ", ";
	if ( operand ) operand->printOneLine( os, indent + 2 );
	os << ") ";
}

//##############################################################################

void LabelNode::print( std::ostream &os, int indent ) const {}

void LabelNode::printOneLine( std::ostream &os, int indent ) const {}

//##############################################################################

ValofExprNode::ValofExprNode( StatementNode *s ): body( s ) {}

ValofExprNode::ValofExprNode( const ValofExprNode &other ) : ExpressionNode( other ), body( maybeClone( body ) ) {
}

ValofExprNode::~ValofExprNode() {
	delete body;
}

void ValofExprNode::print( std::ostream &os, int indent ) const {
	printDesignation( os );
	os << string( indent, ' ' ) << "Valof Expression:" << std::endl;
	get_body()->print( os, indent + 4);
}

void ValofExprNode::printOneLine( std::ostream &, int indent ) const {
	assert( false );
}

Expression *ValofExprNode::build() const {
	return new UntypedValofExpr ( maybeBuild<Statement>(get_body()), maybeBuild< Expression >( get_argName() ) );
}

//##############################################################################

ForCtlExprNode::ForCtlExprNode( ParseNode *init_, ExpressionNode *cond, ExpressionNode *incr ) throw ( SemanticError ) : condition( cond ), change( incr ) {
	if ( init_ == 0 )
		init = 0;
	else {
		DeclarationNode *decl;
		ExpressionNode *exp;

		if (( decl = dynamic_cast<DeclarationNode *>(init_) ) != 0)
			init = new StatementNode( decl );
		else if (( exp = dynamic_cast<ExpressionNode *>( init_)) != 0)
			init = new StatementNode( StatementNode::Exp, exp );
		else
			throw SemanticError("Error in for control expression");
	}
}

ForCtlExprNode::ForCtlExprNode( const ForCtlExprNode &other )
	: ExpressionNode( other ), init( maybeClone( other.init ) ), condition( maybeClone( other.condition ) ), change( maybeClone( other.change ) ) {
}

ForCtlExprNode::~ForCtlExprNode() {
	delete init;
	delete condition;
	delete change;
}

Expression *ForCtlExprNode::build() const {
	// this shouldn't be used!
	assert( false );
	return 0;
}

void ForCtlExprNode::print( std::ostream &os, int indent ) const{
	os << string( indent,' ' ) << "For Control Expression -- :" << endl;

	os << string( indent + 2, ' ' ) << "initialization:" << endl;
	if ( init != 0 )
		init->printList( os, indent + 4 );

	os << string( indent + 2, ' ' ) << "condition: " << endl;
	if ( condition != 0 )
		condition->print( os, indent + 4 );
	os << string( indent + 2, ' ' ) << "increment: " << endl;
	if ( change != 0 )
		change->print( os, indent + 4 );
}

void ForCtlExprNode::printOneLine( std::ostream &, int indent ) const {
	assert( false );
}

//##############################################################################

TypeValueNode::TypeValueNode( DeclarationNode *decl ) : decl( decl ) {
}

TypeValueNode::TypeValueNode( const TypeValueNode &other ) : ExpressionNode( other ), decl( maybeClone( other.decl ) ) {
}

Expression *TypeValueNode::build() const {
	return new TypeExpr( decl->buildType() );
}

void TypeValueNode::print( std::ostream &os, int indent ) const {
	os << std::string( indent, ' ' ) << "Type:";
	get_decl()->print( os, indent + 2);
}

void TypeValueNode::printOneLine( std::ostream &os, int indent ) const {
	os << "Type:";
	get_decl()->print( os, indent + 2);
}


CompoundLiteralNode::CompoundLiteralNode( DeclarationNode *type, InitializerNode *kids ) : type( type ), kids( kids ) {}
CompoundLiteralNode::CompoundLiteralNode( const CompoundLiteralNode &other ) : ExpressionNode( other ), type( other.type ), kids( other.kids ) {}

CompoundLiteralNode::~CompoundLiteralNode() {
	delete kids;
	delete type;
}

CompoundLiteralNode *CompoundLiteralNode::clone() const {
	return new CompoundLiteralNode( *this );
}

void CompoundLiteralNode::print( std::ostream &os, int indent ) const {
	os << string( indent,' ' ) << "CompoundLiteralNode:" << endl;

	os << string( indent + 2, ' ' ) << "type:" << endl;
	if ( type != 0 )
		type->print( os, indent + 4 );

	os << string( indent + 2, ' ' ) << "initialization:" << endl;
	if ( kids != 0 )
		kids->printList( os, indent + 4 );
}

void CompoundLiteralNode::printOneLine( std::ostream &os, int indent ) const {
	os << "( ";
	if ( type ) type->print( os );
	os << ", ";
	if ( kids ) kids->printOneLine( os );
	os << ") ";
}

Expression *CompoundLiteralNode::build() const {
	Declaration * newDecl = maybeBuild<Declaration>(type); // compound literal type
	if ( DeclarationWithType * newDeclWithType = dynamic_cast< DeclarationWithType * >( newDecl ) ) { // non-sue compound-literal type
		return new CompoundLiteralExpr( newDeclWithType->get_type(), maybeBuild<Initializer>(kids) );
	// these types do not have associated type information
	} else if ( StructDecl * newDeclStructDecl = dynamic_cast< StructDecl * >( newDecl )  ) {
		return new CompoundLiteralExpr( new StructInstType( Type::Qualifiers(), newDeclStructDecl->get_name() ), maybeBuild<Initializer>(kids) );
	} else if ( UnionDecl * newDeclUnionDecl = dynamic_cast< UnionDecl * >( newDecl )  ) {
		return new CompoundLiteralExpr( new UnionInstType( Type::Qualifiers(), newDeclUnionDecl->get_name() ), maybeBuild<Initializer>(kids) );
	} else if ( EnumDecl * newDeclEnumDecl = dynamic_cast< EnumDecl * >( newDecl )  ) {
		return new CompoundLiteralExpr( new EnumInstType( Type::Qualifiers(), newDeclEnumDecl->get_name() ), maybeBuild<Initializer>(kids) );
	} else {
		assert( false );
	} // if
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
