# Porting notes for new AST #

## Pointer Types ##
* raw pointer `T*` is used for construction, but not storage
* `ast::ptr_base<T,R>` is a pointer to AST node `T` with reference type `R`
  * specialization: strong pointer `ast::ptr<T>` is used for an ownership relationship
  * specialization: weak pointer `ast::readonly<T>` is used for an observation relationship
* added `ast::ptr_base<T,R>::as<S>()` with same semantics as `dynamic_cast<S*>(p)`
* added `N * ast::ptr_base<N,R>::set_and_mutate( const N * n )`
  * takes ownership of `n`, then returns a mutable version owned by this pointer
  * Some debate on whether this is a good approach:
    * makes an easy path to cloning, which we were trying to eliminate
      * counter-point: these are all mutating clones rather than lifetime-preserving clones, and thus "necessary" (for some definition)
    * existing uses:
      * `VariableExpr::VariableExpr`, `UntypedExpr::createDeref`
        * both involve grabbing a type from elsewhere and making an `lvalue` copy of it
        * could potentially be replaced by a view class something like this:
          ```
          template<unsigned Quals>
          class AddQualifiersType final : public Type {
            readonly<Type> base;
            // ...
          };
          ```
          * requires all `qualifiers` use (and related helpers) to be virtual, non-zero overhead
          * also subtle semantic change, where mutations to the source decl now change the viewing expression

## Visitors ##
* `Visitor` and `Mutator` are combined into a single `ast::Visitor` class
  * Base nodes now override `const Node * accept( Visitor & v ) const = 0` with, e.g. `const Stmt * accept( Visitor & v ) const override = 0`
* `PassVisitor` is replaced with `ast::Pass`

## Structural Changes ##
`CodeLocation` has been devolved from `BaseSyntaxNode` to `ast::ParseNode`
* excludes `ast::Type` from carrying location information
* `CodeLocation` is a mandatory constructor field for `ast::ParseNode`
  * all subclass constructors must fill it; by convention, from their first argument

`N->print(std::ostream&)` is a visitor now
* `Declaration::printShort` is also integrated

`clone` is private to `Node` now
* still needs to be overriden to return appropriate type
  * e.g. `private: virtual Stmt * clone() const override = 0;`
  * because friendship is not inherited, all implementations of clone need
      /// Must be copied in ALL derived classes
      template<typename node_t>
      friend node_t * mutate(const node_t * node);

All leaves of the `Node` inheritance tree are now declared `final`
* e.g. `class CompoundStmt final : public Stmt`
* allows compiler to optimize virtual calls to static calls if given static type

Pulled `FuncSpecifiers`, `StorageClasses`, `CVQualifiers` out of `Type` into their own headers
* Made `BFCommon` a `MakeBitfield` macro in its own header
  * added default and field-init constructors to macro

Prefer move semantics for containers passed to node constructors

## Code Style ##

### Files ###
* Headers have a `.hpp` suffix
* Source code has a `.cpp` suffix
* All source has the project-standard leading and trailing comments
* prefer `#pragma once` over `#ifdef` guards
* namespaces that cover entire files don't get indented
* The general node headers only `#include "Fwd.hpp"` if they can get away with it
  * Anything that needs definitions goes in the .cpp file
  * `Type.hpp` includes `Decl.hpp` so that it knows the `AggregateDecl` subclasses for `ReferenceToType::aggr()` overloads

### Documentation ###
* class, method, and field comments should use doxygen-style `///` prefix
  * should be included on all classes
  * should be included on any method declaration that doesn't have an obvious behaviour from either naming convention (e.g. constructor, print operator, implement visitor pattern) or an inline implementation
* use explanatory comments with `//` wherever appropriate
  * older comments should be maintained in porting process wherever possible

### Naming ###
* Preserve names from previous AST whenever reasonable, and get team consensus on any changes.
* Strong justification required for private fields
  * No `get_` prefix on getters (including for generated fields)
    * exception is `DeclWithType::get_type()`
* Notable changes:
  * for concision and consistency with subclasses:
    * `Declaration` => `ast::Decl`
	* `DeclarationWithType` => `ast::DeclWithType`
	* `Expression` => `ast::Expr`
	* `Initializer` => `ast::Init`
    * `Statement` => `ast::Stmt`
	* any field names should follow a similar renaming
  * because they don't really belong to `Type` (and for consistency with `Linkage::Spec`):
    * `Type::StorageClasses` => `ast::Storage::Classes`
	  * `Type::Extern` etc. => `ast::Storage::Extern` etc.
	* `Type::FuncSpecifiers` => `ast::Function::Specs`
	  * `Type::Inline` etc. => `ast::Function::Inline` etc.
	* `Type::Qualifiers` => `ast::CV::Qualifiers`
	  * `Type::Const` etc. => `ast::CV::Const`
	  * couldn't break name-dependency loop without pulling `Qualifiers` out of `Type`
	* `LinkageSpec::Spec` => `ast::Linkage::Spec`
	  * `LinkageSpec::Mangle` etc. => `ast::Linkage::Mangle` etc.
	  * `LinkageSpec::linkageUpdate` => `ast::Linkage::update`
	  * `LinkageSpec::linkageName` => `ast::Linkage::name`
	  * `LinkageSpec::isMangled(Spec)` etc. => `Spec.is_mangled` etc.
	  * `LinkageSpec::Intrinsic` etc. => `ast::Linkage::Intrinsic` etc.
  * Boolean flags to `SymTab::Mangler::mangle` are now a `SymTab::Mangle::Mode` struct
    * uses `bitfield`
  * Because `Indexer` isn't a terribly evocative name:
    * `SymTab::Indexer` => `ast::SymbolTable`
    * `SymTab/Indexer.{h,cc}` => `AST/SymbolTable.{hpp,cpp}`
    * `WithIndexer` => `WithSymbolTable`
      * `indexer` => `symTab`
    * `IdData::deleteStmt` => `IdData::deleter`
    * `lookupId()` now returns a vector rather than an out-param list
  * To avoid name collisions:
    * `SymTab::Mangler` => `Mangle`
  * `ResolvExpr::TypeEnvironment` => `ast::TypeEnvironment`
    * in `AST/TypeEnvironment.hpp`
* Boolean constructor parameters get replaced with a dedicated flag enum:
  * e.g. `bool isVarLen;` => `enum LengthFlag { FixedLen, VariableLen };` `LengthFlag isVarLen;`
  * field can be *read* in the existing boolean contexts, but requires documentation to write
  * suggest naming all flag enums `FooFlag` to hint at boolean nature

## Specific Nodes ##
`Attribute`
* `parameters` => `params`

`Decl`
* `storageClasses` => `storage`
* `declFromId()` => `fromId()`
  * not 100% sure about the return type here...

`DeclWithType`
* When `SymTab::Validate::Pass2` is rewritten, update comment on `mangleName` with new name of pass
* `get_scopedMangleName()` => `scopedMangleName()`
* `get_type()` now returns `const Type*` so can't be inadvertently mutated
  * still with `get_` name so it doesn't conflict with subclass field names

`ObjectDecl`
* changed constructor parameter order for better defaults
  * allows `newObject` as just default settings

`NamedTypeDecl`
* `parameters` => `params`

`TypeDecl`
* moved `TypeDecl::Kind` to `ast::TypeVar::Kind`

`AggregateDecl`
* `parameters` => `params`

`Expr`
* Merged `inferParams`/`resnSlots` into union, as suggested by comment in old version
  * does imply get_/set_ API, and some care about moving backward
* added constructor that sets result, for benefit of types that set it directly

`ApplicationExpr`
* `function` => `func`

`UntypedExpr`
* `function` => `func`
* removed `begin_args()` in favour of `args.begin()`

`ConstantExpr`
* inlined features of `Constant`, never used elsewhere, so removed `Constant`
  * `Constant Constant::from_int(int)` etc. => `ConstantExpr * ConstantExpr::from_int(CodeLocation, int)`
    * allocates new `ConstantExpr`, consistent with all existing uses

`SizeofExpr`, `AlignofExpr`
* `isType` deprecated in favour of boolean check on `type`
  * all existing uses assume `type` set if true and don't use `expr`

`LogicalExpr`
* un-defaulted constructor parameter determining `&&` or `||`

`CompoundLiteralExpr`
* `initializer` => `init`

`RangeExpr`
* removed `set_low`, `set_high` due to disuse

`TupleIndexExpr`
* removed `set_tuple`, `set_index` due to disuse

`GenericExpr`
* `Association::isDefault` removed: `! type` is equivalent

`StmtExpr`
* `statements` => `stmts`

`Init`
* `bool maybeConstruct` => `enum ConstructFlag { DoConstruct, MaybeConstruct }`

`Label`
* `get_statement()` exclusively used for code location, replaced with `CodeLocation` field

`CaseStmt`
* `_isDefault` has been removed
  * `isDefault` calculates value from `cond`
  * default may not have a condition. I believe case (!default) requires a condition.

`BranchStmt`
* `Type` -> `Kind` and `type` -> `kind`
* Constructors no longer throw SemanticErrorException:
  * `label` constructor claims it is now considered a syntax error, replaced with assert.
  * `computedTarget` constructor assumes `Goto`, other check would have SegFaulted.

`TryStmt`
* `block` -> `body` and `finallyBlock` -> `finally`

`ThrowStmt` `CatchStmt`
* moved `Kind` enums to shared `ast::ExceptionKind` enum

`FinallyStmt`
* `block` -> `body`

`CompoundStmt`
* Still a `std::list` for children, rather than `std::vector`
  * allows more-efficient splicing for purposes of later code generation

`Type`
* `CV::Qualifiers` moved to end of constructor parameter list, defaulted to `{}`
  * removed getter, setter in favour of public `qualifiers` field
  * `ReferenceToType` puts a defaulted list of attributes after qualifiers
* `forall` field split off into `ParameterizedType` subclass
  * any type that needs it can inherit from `ParameterizedType`
    * currently `FunctionType`, `ReferenceToType`
* `get_qualifiers()` replaced with accessor `qualifiers()` and mutator `set_qualifiers()`
  * `get_const()` etc. replaced with `is_const()` etc. variants
* `referenceDepth()` now returns `unsigned` rather than `int`
* A number of features only supported on aggregates pushed down to `ReferenceToType`:
  * `attributes`: per docs [1] GCC only supports type attributes on aggregates and typedefs
    * suggest adding a `TypeWithAttributes` wrapper type if this proves insufficient
  * `getAggr()` => `aggr()`
    * also now returns `const AggregateDecl *`
* `genericSubstitution()` moved to own visitor in `AST/GenericSubstitution.hpp`
  * subsumes old `makeGenericSubstitution()`

`BasicType`
* **TODO** move `kind`, `typeNames` into code generator

`ReferenceToType`
* deleted `get_baseParameters()` from children
  * replace with `aggr() ? aggr()->params : nullptr`
* `parameters` => `params`
* hoisted `lookup` implementation into parent, made non-virtual
  * also changed to return vector rather than filling; change back if any great win for reuse
* `baseStruct` etc. renamed to `base`

`PointerType`/`ArrayType`
* `is_array()` => `isArray()`
* `bool isVarLen;` => `enum LengthFlag { FixedLen, VariableLen }; LengthFlag isVarLen;`
* `bool isStatic;` => `enum DimensionFlag { DynamicDim, StaticDim }; DimensionFlag isStatic;`

`FunctionType`
* `returnVals` => `returns`
* `parameters` => `params`
* `bool isVarArgs;` => `enum ArgumentFlag { FixedArgs, VariableArgs }; ArgumentFlag isVarArgs;`

`TypeInstType`
* `bool isFtype` => `TypeVar::Kind kind`

`TypeofType`
* `bool is_basetypeof` => `enum Kind { Typeof, Basetypeof } kind;`

`TupleType`
* removed `value_type` typedef due to likely error
  * if readded, should be `const Type *`

`AttrType`
* did not port due to deprecation of feature
  * feature is `type@thing` e.g. `int@MAX`

`referenceToRvalueConversion`
* now returns `const Expr *` rather than mutating argument

`printAssertionSet`, `printOpenVarSet`
* `ostream &` now first argument, for consistency

`EqvClass`
* `type` => `bound`

`TypeEnvironment`
* `makeSubstitution()` => `writeToSubstitution()`
* `isEmpty()` => `empty()`
* removed `clone()` in favour of explicit copies

`occurs`
* moved to be helper function in `TypeEnvironment.cpp` (its only use)

`WidenMode`
* changed `widenFirst`, `widenSecond` => `first`, `second`
* changed `WidenMode widenMode` => `WidenMode widen`

`Alternative` => `Candidate`
* `openVars` => `open`

`ExplodedActual` => `ExplodedArg`
* `ExplodedActual.h` => `ExplodedArg.hpp`

`polyCost`
* switched order of `env`, `symtab` parameters for better consistency

`findMinCost`
* pulled out conversion cost promotion into separate `promoteCvtCost` function

`resolveAssertions` => `satisfyAssertions`
* `ResolveAssertions.h` => `SatisfyAssertions.hpp`
* `Resn*` => `Sat*`

[1] https://gcc.gnu.org/onlinedocs/gcc-9.1.0/gcc/Type-Attributes.html#Type-Attributes

