//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Explode.h --
//
// Author           : Rob Schluntz
// Created On       : Wed Nov 9 13:12:24 2016
// Last Modified By : Peter A. Buhr
// Last Modified On : Sat Jul 22 09:55:16 2017
// Update Count     : 3
//

#pragma once

#include <iterator>                  // for back_inserter, back_insert_iterator

#include "ResolvExpr/Alternative.h"  // for Alternative, AltList
#include "SynTree/Expression.h"      // for Expression, UniqueExpr, AddressExpr
#include "SynTree/Type.h"            // for TupleType, Type
#include "Tuples.h"                  // for maybeImpure

namespace SymTab {
class Indexer;
}  // namespace SymTab

namespace Tuples {
	/// helper function used by explode to properly distribute
	/// '&' across a tuple expression
	Expression * distributeAddr( Expression * expr );

	/// helper function used by explode
	template< typename OutputIterator >
	void explodeUnique( Expression * expr, const ResolvExpr::Alternative & alt, const SymTab::Indexer & indexer, OutputIterator out, bool isTupleAssign ) {
		if ( isTupleAssign ) {
			// tuple assignment needs AddressExprs to be recursively exploded to easily get at all of the components
			if ( AddressExpr * addrExpr = dynamic_cast< AddressExpr * >( expr ) ) {
				ResolvExpr::AltList alts;
				explodeUnique( addrExpr->get_arg(), alt, indexer, back_inserter( alts ), isTupleAssign );
				for ( ResolvExpr::Alternative & alt : alts ) {
					// distribute '&' over all components
					alt.expr = distributeAddr( alt.expr );
					*out++ = alt;
				}
				// in tuple assignment, still need to handle the other cases, but only if not already handled here (don't want to output too many alternatives)
				return;
			}
		}
		Type * res = expr->get_result();
		if ( TupleType * tupleType = dynamic_cast< TupleType * > ( res ) ) {
			if ( TupleExpr * tupleExpr = dynamic_cast< TupleExpr * >( expr ) ) {
				// can open tuple expr and dump its exploded components
				for ( Expression * expr : tupleExpr->get_exprs() ) {
					explodeUnique( expr, alt, indexer, out, isTupleAssign );
				}
			} else {
				// tuple type, but not tuple expr - recursively index into its components
				Expression * arg = expr->clone();
				if ( Tuples::maybeImpure( arg ) && ! dynamic_cast< UniqueExpr * >( arg ) ) {
					// expressions which may contain side effects require a single unique instance of the expression.
					arg = new UniqueExpr( arg );
				}
				for ( unsigned int i = 0; i < tupleType->size(); i++ ) {
					TupleIndexExpr * idx = new TupleIndexExpr( arg->clone(), i );
					explodeUnique( idx, alt, indexer, out, isTupleAssign );
					delete idx;
				}
				delete arg;
			}
		} else {
			// atomic (non-tuple) type - output a clone of the expression in a new alternative
			*out++ = ResolvExpr::Alternative( expr->clone(), alt.env, alt.cost, alt.cvtCost );
		}
	}

	/// expands a tuple-valued alternative into multiple alternatives, each with a non-tuple-type
	template< typename OutputIterator >
	void explode( const ResolvExpr::Alternative &alt, const SymTab::Indexer & indexer, OutputIterator out, bool isTupleAssign = false ) {
		explodeUnique( alt.expr, alt, indexer, out, isTupleAssign );
	}

	// explode list of alternatives
	template< typename AltIterator, typename OutputIterator >
	void explode( AltIterator altBegin, AltIterator altEnd, const SymTab::Indexer & indexer, OutputIterator out, bool isTupleAssign = false ) {
		for ( ; altBegin != altEnd; ++altBegin ) {
			explode( *altBegin, indexer, out, isTupleAssign );
		}
	}

	template< typename OutputIterator >
	void explode( const ResolvExpr::AltList & alts, const SymTab::Indexer & indexer, OutputIterator out, bool isTupleAssign = false ) {
		explode( alts.begin(), alts.end(), indexer, out, isTupleAssign );
	}
} // namespace Tuples

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
