//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Mangler.h --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 21:44:03 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Wed Aug 19 15:48:46 2015
// Update Count     : 14
//

#ifndef MANGLER_H
#define MANGLER_H

#include <map>                // for map, map<>::value_compare
#include <sstream>            // for ostringstream
#include <string>             // for string
#include <utility>            // for pair

#include "SynTree/SynTree.h"  // for Types
#include "SynTree/Visitor.h"  // for Visitor, maybeAccept

namespace SymTab {
	/// Mangles names to a unique C identifier
	class Mangler : public Visitor {
	  public:
		/// Mangle syntax tree object; primary interface to clients
		template< typename SynTreeClass >
	    static std::string mangle( SynTreeClass *decl, bool mangleOverridable = true, bool typeMode = false );
		/// Mangle a type name; secondary interface
		static std::string mangleType( Type* ty );

		virtual void visit( ObjectDecl *declaration );
		virtual void visit( FunctionDecl *declaration );
		virtual void visit( TypeDecl *declaration );

		virtual void visit( VoidType *voidType );
		virtual void visit( BasicType *basicType );
		virtual void visit( PointerType *pointerType );
		virtual void visit( ArrayType *arrayType );
		virtual void visit( FunctionType *functionType );
		virtual void visit( StructInstType *aggregateUseType );
		virtual void visit( UnionInstType *aggregateUseType );
		virtual void visit( EnumInstType *aggregateUseType );
		virtual void visit( TypeInstType *aggregateUseType );
		virtual void visit( TupleType *tupleType );
		virtual void visit( VarArgsType *varArgsType );
		virtual void visit( ZeroType *zeroType );
		virtual void visit( OneType *oneType );

		std::string get_mangleName() { return mangleName.str(); }
	  private:
		std::ostringstream mangleName;  ///< Mangled name being constructed
		typedef std::map< std::string, std::pair< int, int > > VarMapType;
		VarMapType varNums;             ///< Map of type variables to indices
		int nextVarNum;                 ///< Next type variable index
		bool isTopLevel;                ///< Is the Mangler at the top level
		bool mangleOverridable;         ///< Specially mangle overridable built-in methods
		bool typeMode;                  ///< Produce a unique mangled name for a type

		Mangler( bool mangleOverridable, bool typeMode );
		Mangler( const Mangler & );

		void mangleDecl( DeclarationWithType *declaration );
		void mangleRef( ReferenceToType *refType, std::string prefix );
		void mangleGenericRef( ReferenceToType *refType, std::string prefix );

		void printQualifiers( Type *type );
	}; // Mangler

	template< typename SynTreeClass >
	std::string Mangler::mangle( SynTreeClass *decl, bool mangleOverridable, bool typeMode ) {
		Mangler mangler( mangleOverridable, typeMode );
		maybeAccept( decl, mangler );
		return mangler.get_mangleName();
	}
} // SymTab

#endif // MANGLER_H

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
